package rsserver

import (
	"a.yandex-team.ru/crypta/lib/go/yt/kv"
	"context"
	"encoding/json"
	"fmt"
	"strconv"
	"time"
)

const (
	reports = "reports"
)

type ReportCounts struct {
	OkCount      int `json:"ok_count"`
	VersionCount int `json:"version_count"`
}

type ReportsClient interface {
	ReportVersion(ctx context.Context, env string, resource string, version uint64, remoteAddr string) error
	ReportOk(ctx context.Context, env string, resource string, version uint64, remoteAddr string) error
	GetReportCounts(ctx context.Context, env string, resource string, version uint64) (ReportCounts, error)
}

type reportsClient struct {
	client        kv.Client
	reportTimeout time.Duration
}

func (client *reportsClient) ReportVersion(ctx context.Context, env string, resource string, version uint64, remoteAddr string) error {
	ctx, cancel := context.WithTimeout(ctx, client.reportTimeout)
	defer cancel()

	return writeJSON(ctx, client.client, getReportsVersionKey(env, resource, version, remoteAddr), time.Now().Unix())
}

func (client *reportsClient) ReportOk(ctx context.Context, env string, resource string, version uint64, remoteAddr string) error {
	ctx, cancel := context.WithTimeout(ctx, client.reportTimeout)
	defer cancel()

	return writeJSON(ctx, client.client, getReportsOkKey(env, resource, version, remoteAddr), time.Now().Unix())
}

func (client *reportsClient) GetReportCounts(ctx context.Context, env string, resource string, version uint64) (ReportCounts, error) {
	reportsOk, err := client.selectReportsWithPrefix(ctx, getReportsOkKeyPrefix(env, resource, version))
	if err != nil {
		return ReportCounts{}, err
	}

	reportsVersion, err := client.selectReportsWithPrefix(ctx, getReportsVersionKeyPrefix(env, resource, version))
	if err != nil {
		return ReportCounts{}, err
	}

	return ReportCounts{
		OkCount:      len(reportsOk),
		VersionCount: len(reportsVersion),
	}, nil
}

func NewReportsClient(client kv.Client, reportTimeout time.Duration) ReportsClient {
	return &reportsClient{client: client, reportTimeout: reportTimeout}
}

func getReportsOkKeyPrefix(env string, resource string, version uint64) string {
	return getKey(reports, "ok", env, resource, strconv.FormatUint(version, 10))
}

func getReportsOkKey(env string, resource string, version uint64, clientID string) string {
	return getKey(getReportsOkKeyPrefix(env, resource, version), clientID)
}

func getReportsVersionKeyPrefix(env string, resource string, version uint64) string {
	return getKey(reports, "version", env, resource, strconv.FormatUint(version, 10))
}

func getReportsVersionKey(env string, resource string, version uint64, clientID string) string {
	return getKey(getReportsVersionKeyPrefix(env, resource, version), clientID)
}

func (client *reportsClient) selectReportsWithPrefix(ctx context.Context, prefix string) (map[string]int64, error) {
	result := make(map[string]int64)

	records, err := client.client.Select(ctx, fmt.Sprintf("WHERE is_prefix(\"%s\", key)", prefix))
	if err != nil {
		return nil, err
	}

	for key, value := range records {
		var buf int64
		err := json.Unmarshal([]byte(value), &buf)
		if err != nil {
			return nil, err
		}
		result[key] = buf
	}

	return result, err
}
