package rsserver

import (
	"a.yandex-team.ru/library/go/core/metrics"
	"context"
	"fmt"
	"github.com/go-chi/chi/v5"
	"github.com/go-chi/chi/v5/middleware"
	"log"
	"net"
	"net/http"
	"strconv"
	"strings"
	"time"
)

type contextKey string

const (
	metricsCtxKey = contextKey("metricsCtx")
)

type handler struct {
	dbClient          ReadOnlyDBClient
	resources         Resources
	fileClient        FileClient
	reportsClient     ReportsClient
	httpCodes         metrics.CounterVec
	resourceHTTPCodes metrics.CounterVec
}

type resourceHandlerContext struct {
	Env      string
	Resource string
}

type metricsContext struct {
	Cmd                    string
	ResourceHandlerContext *resourceHandlerContext
}

type resourceHandler func(http.ResponseWriter, *http.Request, resourceHandlerContext)

type versionHandler func(http.ResponseWriter, *http.Request, resourceHandlerContext, uint64)

func CreateServer(dbClient ReadOnlyDBClient, resources Resources, fileClient FileClient, reportsClient ReportsClient, httpConfig HTTPConfig, stats metrics.Registry) *http.Server {
	router := chi.NewRouter()
	handler := handler{
		dbClient,
		resources,
		fileClient,
		reportsClient,
		stats.CounterVec("http_codes", []string{"cmd", "code"}),
		stats.CounterVec("resource_http_codes", []string{"cmd", "code", "env", "resource"}),
	}

	router.Use(middleware.RequestID)
	router.Use(middleware.Logger)
	router.Use(handler.addMetrics)
	router.Use(middleware.Timeout(time.Duration(httpConfig.TimeoutSec) * time.Second))
	router.Use(middleware.Recoverer)

	router.Get("/ping", handler.ping)
	router.Get("/version/{dc}/{resource}", handler.newResourceHandler(handler.getVersion))
	router.Get("/get/{dc}/{resource}/{version:[0-9]+}", handler.newVersionHandler(handler.getResource))
	router.Get("/report_ok/{dc}/{resource}/{version:[0-9]+}", handler.newVersionHandler(handler.reportOk))
	router.Get("/get_report_counts/{dc}/{resource}/{version:[0-9]+}", handler.newVersionHandler(handler.getReportCounts))
	router.NotFound(handler.notFound)

	return &http.Server{
		Addr:    fmt.Sprintf(":%v", httpConfig.Port),
		Handler: router,
	}
}

func (handler *handler) newResourceHandler(next resourceHandler) http.HandlerFunc {
	fn := func(writer http.ResponseWriter, request *http.Request) {
		metricsCtx := request.Context().Value(metricsCtxKey).(*metricsContext)

		resourceCtx := &resourceHandlerContext{
			Env:      "unknown",
			Resource: "unknown",
		}
		metricsCtx.ResourceHandlerContext = resourceCtx

		dc := chi.URLParam(request, "dc")

		env, err := handler.dbClient.GetEnv(request.Context(), dc)
		if err != nil || env == nil {
			replyNotFound(writer, fmt.Sprintf("Env for %s not found: %v", dc, err))
			return
		}
		resourceCtx.Env = *env

		resourceName := chi.URLParam(request, "resource")
		_, ok := handler.resources[resourceName]

		if !ok {
			replyNotFound(writer, fmt.Sprintf("Resource %v not found", resourceName))
			return
		}
		resourceCtx.Resource = resourceName

		next(writer, request, *resourceCtx)
	}
	return fn
}

func (handler *handler) newVersionHandler(next versionHandler) http.HandlerFunc {
	fn := func(writer http.ResponseWriter, request *http.Request, resourceCtx resourceHandlerContext) {
		versionStr := chi.URLParam(request, "version")
		version, err := strconv.ParseUint(versionStr, 10, 64)
		if err != nil {
			replyError(writer, fmt.Sprintf("Bad version format %v: %v", versionStr, err), http.StatusBadRequest)
			return
		}

		next(writer, request, resourceCtx, version)
	}
	return handler.newResourceHandler(fn)
}

func (handler *handler) addMetrics(next http.Handler) http.Handler {
	fn := func(writer http.ResponseWriter, request *http.Request) {
		wrappedWriter := middleware.NewWrapResponseWriter(writer, request.ProtoMajor)

		metricsCtx := &metricsContext{
			Cmd: strings.Split(request.URL.Path, "/")[1],
		}
		ctx := context.WithValue(request.Context(), metricsCtxKey, metricsCtx)

		next.ServeHTTP(wrappedWriter, request.WithContext(ctx))

		labels := map[string]string{
			"cmd":  metricsCtx.Cmd,
			"code": strconv.FormatInt(int64(wrappedWriter.Status()), 10),
		}

		if metricsCtx.ResourceHandlerContext != nil {
			labels["resource"] = metricsCtx.ResourceHandlerContext.Resource
			labels["env"] = metricsCtx.ResourceHandlerContext.Env
			handler.resourceHTTPCodes.With(labels).Inc()
		} else {
			handler.httpCodes.With(labels).Inc()
		}
	}
	return http.HandlerFunc(fn)
}

func (handler *handler) ping(writer http.ResponseWriter, request *http.Request) {
	resourceIDs, err := handler.dbClient.GetPublicResources(request.Context())
	if err != nil {
		replyInternalError(writer, err)
		return
	}

	for _, resourceID := range resourceIDs {
		if !handler.fileClient.IsPresent(resourceID.Name, resourceID.Version) {
			replyInternalError(writer, fmt.Errorf("resource %v %v is not ready yet", resourceID.Name, resourceID.Version))
			return
		}
	}
	replyOk(writer, "OK")
}

func (handler *handler) getVersion(writer http.ResponseWriter, request *http.Request, resourceCtx resourceHandlerContext) {
	version, err := handler.dbClient.GetLatestResourceVersion(request.Context(), resourceCtx.Env, resourceCtx.Resource)
	if err != nil {
		replyInternalError(writer, err)
		return
	}
	if version == nil {
		replyNotFound(writer, fmt.Sprintf("Resource %s for %s not found", resourceCtx.Resource, resourceCtx.Env))
		return
	}

	err = handler.reportsClient.ReportVersion(request.Context(), resourceCtx.Env, resourceCtx.Resource, *version, getRemoteAddr(request))
	if err != nil {
		log.Printf("failed to report version: %v", err)
	}
	replyOk(writer, map[string]interface{}{"version": version})
}

func (handler *handler) getResource(writer http.ResponseWriter, request *http.Request, resourceCtx resourceHandlerContext, version uint64) {
	actualVersion, err := handler.dbClient.GetLatestResourceVersion(request.Context(), resourceCtx.Env, resourceCtx.Resource)
	if err != nil {
		replyInternalError(writer, err)
		return
	}
	if actualVersion == nil || *actualVersion != version {
		replyNotFound(writer, fmt.Sprintf("Resource %s with version %v for %s not found", resourceCtx.Resource, version, resourceCtx.Env))
		return
	}

	filePath := handler.fileClient.GetResourceFilename(resourceCtx.Resource, version)

	http.ServeFile(writer, request, filePath)
}

func (handler *handler) reportOk(writer http.ResponseWriter, request *http.Request, resourceCtx resourceHandlerContext, version uint64) {
	err := handler.reportsClient.ReportOk(request.Context(), resourceCtx.Env, resourceCtx.Resource, version, getRemoteAddr(request))
	if err != nil {
		log.Printf("failed to report ok: %v", err)
	}

	replyOk(writer, "OK")
}

func (handler *handler) getReportCounts(writer http.ResponseWriter, request *http.Request, resourceCtx resourceHandlerContext, version uint64) {
	counts, err := handler.reportsClient.GetReportCounts(request.Context(), resourceCtx.Env, resourceCtx.Resource, version)
	if err != nil {
		replyInternalError(writer, err)
		return
	}
	replyOk(writer, counts)
}

func (handler *handler) notFound(writer http.ResponseWriter, request *http.Request) {
	metricsCtx := request.Context().Value(metricsCtxKey).(*metricsContext)
	metricsCtx.Cmd = "unknown"
	http.NotFound(writer, request)
}

func getRemoteAddr(request *http.Request) string {
	ip := request.Header.Get("x-real-ip")
	if ip != "" {
		return ip
	}

	ip, _, err := net.SplitHostPort(request.RemoteAddr)
	if err != nil {
		return request.RemoteAddr
	}
	return ip
}
