import logging

import retry

logger = logging.getLogger(__name__)


class StateSynchronizer(object):
    def __init__(self, instance_id, resources, file_client, db, sandbox, retry_opts=None):
        self.instance_id = instance_id
        self.resources = resources
        self.file_client = file_client
        self.db = db
        self.sandbox = sandbox
        self.retry_opts = retry_opts or {}

    def sync(self):
        self.file_client.init(self.resources)
        retry.retry_call(self._sync, **self.retry_opts)

    def _sync(self):
        with self.db.transaction():
            self.db.register_instance(self.instance_id)

            resources = self.db.get_resources_to_download()
            for name, id_ in resources:
                if not self.file_client.is_present(name, id_):
                    target_path = self.file_client.get_resource_filename(name, id_)
                    self.sandbox.load_resource(id_, target_path, resource_path=self.resources[name].path, bundle_file=self.resources[name].bundle_file)

            self.db.set_instance_resources(self.instance_id, resources)

        self.file_client.remove_extra(resources)
