import logging

import retry

from crypta.lib.python.rtmr import api
from crypta.utils.rtmr_resource_service.lib import consts

logger = logging.getLogger(__name__)


class YtStateUpdater(object):
    def __init__(self, resources, db, reports_client, sandbox, cluster_envs, report_ttl, instance_ttl, retry_opts=None):
        self.resources = resources
        self.db = db
        self.reports_client = reports_client
        self.sandbox = sandbox
        self.cluster_envs = cluster_envs
        self.report_ttl = report_ttl
        self.instance_ttl = instance_ttl
        self.retry_opts = retry_opts or {}

    def update(self):
        for func in (
            self.expire_instances,
            self.update_resources,
            self.update_cluster_envs,
            self.expire_reports,
        ):
            retry.retry_call(func, **self.retry_opts)

    def expire_instances(self):
        with self.db.transaction():
            self.db.expire_instances(self.instance_ttl)

    def update_resources(self):
        sandbox_versions = self.get_sandbox_versions()
        with self.db.transaction():
            self.db.lock_instances()

            yt_versions = self.get_yt_versions()
            self.set_resources_to_download(sandbox_versions, yt_versions)

            common_present_resources = self.get_common_present_resources()
            self.update_versions(sandbox_versions, common_present_resources)

    def update_cluster_envs(self):
        with self.db.transaction():
            self.set_cluster_env(self.cluster_envs["testing_cluster"], consts.TESTING)

            for cluster in self.cluster_envs["clusters"]:
                rtmr_client = api.Client("{}.search.yandex.net".format(cluster))
                try:
                    operation_config = retry.retry_call(
                        rtmr_client.get_operation_config,
                        fargs=(self.cluster_envs["tracked_operation"],),
                        **self.retry_opts
                    )
                    env = consts.STABLE if operation_config.Attrs.Enabled else consts.PRESTABLE
                    self.set_cluster_env(cluster, env)
                except Exception:
                    logger.exception("Failed to set RTMR cluster env for %s", cluster)
                    self.set_cluster_env(cluster, consts.STABLE)

    def set_cluster_env(self, cluster, env):
        logger.info("Setting %s to %s", cluster, env)
        self.db.set_env(cluster, env)

    def expire_reports(self):
        with self.reports_client.transaction():
            self.reports_client.expire_reports(self.report_ttl)

    def get_sandbox_versions(self):
        return self._get_versions(lambda release_type, resource: self.sandbox.get_last_released_resource_id(resource.resource_type, release_type))

    def get_yt_versions(self):
        return self._get_versions(lambda release_type, resource: self.db.get_latest_resource_version(release_type, resource.name))

    def _get_versions(self, getter):
        result = []

        for resource in self.resources.values():
            for release_type in consts.ENVS:
                version = getter(release_type, resource)
                if version is not None:
                    result.append((resource.name, release_type, version))

        return result

    def set_resources_to_download(self, sandbox_versions, yt_versions):
        self.db.set_resources_to_download(list(self.versions_to_set(sandbox_versions) | self.versions_to_set(yt_versions)))

    @staticmethod
    def versions_to_set(versions):
        return set((name, version) for name, _, version in versions)

    def get_common_present_resources(self):
        instances = self.db.get_instances()

        if not instances:
            return set()

        return set.intersection(*(set(self.db.get_instance_resources(instance)) for instance in instances))

    def update_versions(self, sandbox_versions, common_present_resources):
        public_resources = [
            (resource_name, release_type, version)
            for resource_name, release_type, version in sandbox_versions
            if (resource_name, version) in common_present_resources
        ]

        for resource_name, release_type, version in public_resources:
            self.db.set_latest_resource_version(release_type, resource_name, version)

        self.db.set_public_resources(public_resources)
