import json
import time

from crypta.utils.rtmr_resource_service.lib import db_utils


INSTANCES = "instances"
RESOURCES_TO_DOWNLOAD = "resources_to_download"
PUBLIC_RESOURCES = "public_resources"


class DbClient(object):
    def __init__(self, kv_client):
        self.kv_client = kv_client

    def transaction(self):
        return self.kv_client.transaction()

    def write(self, key, value):
        self.kv_client.write(key, json.dumps(value))

    def lookup(self, key, default=None):
        value = self.kv_client.lookup(key)
        return default if value is None else json.loads(value)

    def get_latest_resource_version(self, env, resource_name):
        return self.lookup(_resource_key(env, resource_name))

    def set_latest_resource_version(self, env, resource_name, version):
        return self.write(_resource_key(env, resource_name), version)

    def get_env(self, cluster):
        return self.lookup(_env_key(cluster))

    def set_env(self, cluster, env):
        return self.write(_env_key(cluster), env)

    def get_instances(self):
        return set(self.lookup(INSTANCES, {}).keys())

    def lock_instances(self):
        self.write(INSTANCES, self.lookup(INSTANCES))

    def register_instance(self, instance):
        instances = self.lookup(INSTANCES, {})
        instances[instance] = time.time()
        self.write(INSTANCES, instances)

    def expire_instances(self, ttl):
        deadline = time.time() - ttl.total_seconds()
        instances = self.lookup(INSTANCES) or {}
        to_expire = [k for k in instances if instances[k] <= deadline]

        for key in to_expire:
            del instances[key]
            self.kv_client.delete(_instance_key(key))

        self.write(INSTANCES, instances)

    def get_instance_resources(self, instance):
        return set(tuple(x) for x in self.lookup(_instance_key(instance), []))

    def set_instance_resources(self, instance, resources):
        self.write(_instance_key(instance), list(resources))

    def get_resources_to_download(self):
        return set(tuple(x) for x in self.lookup(RESOURCES_TO_DOWNLOAD, []))

    def set_resources_to_download(self, resources):
        self.write(RESOURCES_TO_DOWNLOAD, list(resources))

    def get_public_resources(self):
        return set(tuple(x) for x in self.lookup(PUBLIC_RESOURCES, []))

    def set_public_resources(self, resources):
        self.write(PUBLIC_RESOURCES, list(resources))


def _resource_key(env, resource_name):
    return db_utils.join("resource", env, resource_name)


def _env_key(cluster):
    return db_utils.join("env", cluster)


def _instance_key(instance):
    return db_utils.join("instance", instance)
