import collections
import json
import time

import retry

from crypta.utils.rtmr_resource_service.lib import db_utils
import six

Counts = collections.namedtuple("Counts", ["version", "ok"])

REPORTS = "reports"


def _retry_and_suppress_errors(f):
    def wrapper(self, *args, **kwargs):
        try:
            retry.retry_call(
                f,
                fargs=(self,) + args,
                fkwargs=kwargs,
                **self.retry_opts
            )
        except Exception:
            pass

    return wrapper


class ReportsClient(object):
    def __init__(self, kv_client, retry_opts):
        self.kv_client = kv_client
        self.retry_opts = retry_opts

    def transaction(self):
        return self.kv_client.transaction()

    def write(self, key, value):
        self.kv_client.write(key, json.dumps(value))

    @_retry_and_suppress_errors
    def report_version(self, env, resource_name, version, client_id):
        self._report(_reports_version_key(env, resource_name, version, client_id))

    @_retry_and_suppress_errors
    def report_ok(self, env, resource_name, version, client_id):
        self._report(_reports_ok_key(env, resource_name, version, client_id))

    def _select_reports_with_prefix(self, prefix):
        return {
            k[len(prefix):]: float(v)
            for k, v in six.iteritems(self.kv_client.select('WHERE is_prefix("{}", key)'.format(prefix)))
        }

    def get_report_counts(self, env, resource, version):
        reports_ok = self._select_reports_with_prefix(_reports_ok_key_prefix(env, resource, version))
        reports_version = self._select_reports_with_prefix(_reports_version_key_prefix(env, resource, version))

        version_count = len(reports_version)
        ok_count = len(reports_ok)

        return Counts(version_count, ok_count)

    def expire_reports(self, report_ttl):
        deadline = time.time() - report_ttl.total_seconds()
        reports = self._select_reports_with_prefix(REPORTS)
        to_expire = [REPORTS + k for k in reports if reports[k] <= deadline]
        self.kv_client.delete_many(to_expire)

    def _report(self, key):
        self.write(key, time.time())


def _reports_ok_key_prefix(env, resource_name, version):
    return db_utils.join(REPORTS, "ok", env, resource_name, version)


def _reports_ok_key(env, resource_name, version, client_id):
    prefix = _reports_ok_key_prefix(env, resource_name, version)
    return db_utils.join(prefix, client_id)


def _reports_version_key_prefix(env, resource_name, version):
    return db_utils.join(REPORTS, "version", env, resource_name, version)


def _reports_version_key(env, resource_name, version, client_id):
    prefix = _reports_version_key_prefix(env, resource_name, version)
    return db_utils.join(prefix, client_id)
