import datetime

import yt.wrapper as yt

from crypta.lib.python.solomon.proto import alert_pb2
from crypta.lib.python.spine.consts import environment
from crypta.lib.python.spine.consts.yt_proxy import YtProxy
from crypta.lib.python.spine.juggler import (
    consts,
    juggler_check_generator,
)
from crypta.lib.python.spine.juggler.flap_detector_params import FlapDetectorParams
from crypta.lib.python.spine.solomon import solomon_check_generator
from crypta.lib.python.spine.yt import (
    yt_replicated_table,
)


def add_yt_replicated_table(juggler):
    root_path = "//home/crypta/production/resource_service"
    master = yt_replicated_table.Master(
        name="markov",
        proxy=YtProxy.markov,
        path=yt.ypath_join(root_path, "master"),
        expected_attributes={
            "tablet_cell_bundle": "crypta-cm",
            "replicated_table_options": {
                "enable_replicated_table_tracker": True,
                "max_sync_replica_count": 1,
                "tablet_cell_bundle_name_failure_interval": 60000,
                "tablet_cell_bundle_name_ttl": 300000,
                "sync_replica_lag_threshold": 600000,
            },
        },
        replication_lag_threshold=datetime.timedelta(seconds=60),
        sync_count=1,
    )

    replicas = [
        yt_replicated_table.Replica(
            name=proxy.split(".", 1)[0],
            proxy=proxy,
            path=yt.ypath_join(root_path, "replica"),
            expected_attributes={
                "tablet_cell_bundle": "crypta-cm",
                "in_memory_mode": "uncompressed",
            },
        )
        for proxy in [YtProxy.seneca_vla, YtProxy.seneca_sas]
    ]

    return yt_replicated_table.ReplicatedTableConfigRegistry(juggler, yt_replicated_table.ReplicatedTable(master, replicas))


def get_registry():
    base_juggler_gen = juggler_check_generator.JugglerCheckGenerator(tags=["crypta-idfy-telegram-alert", "crypta-rt-graph"])

    deploy_group = "crypta@stage=crypta-resource-service-production"
    juggler = base_juggler_gen.clone(
        host=deploy_group,
        warn_limit=0,
        crit_limit=1,
        child_group=deploy_group,
        child_group_type=consts.GroupType.deploy,
    )

    add_yt_replicated_table(juggler)

    flap_detector_params = FlapDetectorParams(datetime.timedelta(minutes=10), datetime.timedelta(minutes=30))

    juggler.icmpping().add_flap_detector(flap_detector_params)
    juggler.http("api_alive", path="/ping", port=80, is_unreachable=False).add_flap_detector(flap_detector_params)
    juggler.http("stats_alive", path="/ping", port=4242, is_unreachable=False).add_flap_detector(flap_detector_params)

    solomon = solomon_check_generator.ServiceSolomonCheckGenerator(
        juggler,
        project="crypta_resource_service",
        service="stats",
        cluster=environment.PRODUCTION,
    )

    for code, threshold in (("4??", 0.1), ("5??", 1)):
        solomon.get_sensor({"code": code, "env": "prestable|stable"}).create_threshold_check(
            aggregation=alert_pb2.AVG,
            predicate=alert_pb2.GT,
            threshold=threshold,
            period=datetime.timedelta(minutes=5),
            description=code + " codes: {{ pointValue }}",
            juggler_service="{}_codes".format(code).replace("?", "X")
        ).add_nodata_mode("force_ok").add_flap_detector(FlapDetectorParams(datetime.timedelta(minutes=60), datetime.timedelta(minutes=180)))

    for service in ("resource_service_releaser", "resource_service_state_synchronizer", "resource_service_yt_state_updater"):
        juggler.periodic_task(service, crit_time=datetime.timedelta(hours=2)).add_flap_detector(flap_detector_params)

    return base_juggler_gen
