#!/usr/bin/python
# -*- coding: utf-8 -*-
import os
import re
import argparse
from contextlib import contextmanager
from paramiko.client import SSHClient

from crypta_vault_client import CryptaVaultClient
from vault_secret import VaultSecret

parser = argparse.ArgumentParser(description='Script to migrate from secdist to Yandex Vault')
parser.add_argument('-d', '--secrets_dir', type=str, dest="secrets_dir", default='',
                    help='Relative path with desired secrets directory')
parser.add_argument('--dry-run', type=bool, const=True, dest='dry_run', default=False, nargs='?',
                    help="Dry run mode, secrets won't be added to Vault")
parser.add_argument('--production', type=bool, const=True, default=False, dest='is_production', nargs='?',
                    help='Upload secrets to production Vault, yav-test is used by default')
parser.add_argument('-e', '--expression', type=str, default=None, dest='expression',
                    help='Search Regex to match particular keys')
parser.add_argument('-x', '--exclude-dirs', type=str, default=None, dest='exclude_dirs',
                    help='Exclude directories from transferring to Yandex Vault, use comma to separate')

args = parser.parse_args()


SECDIST_HOST = 'secdist.yandex.net'
PROJECTS_BASEPATH = '/repo/projects/crypta'
DRY_RUN = False
PRODUCTION = False
EXPRESSION = ''
EXCLUDE_DIRS = None

client = SSHClient()
client.load_system_host_keys()

CRYPTA_VAULT_CLIENT = CryptaVaultClient(PRODUCTION)


@contextmanager
def read_secret(secret_path, normalize=False):

    if not normalize:
        normalized_path = secret_path
    else:
        path_parts = secret_path.split('/')
        normalized_name = re.sub('[^A-Za-z0-9]+', '_', path_parts[-1])
        path_parts[-1] = normalized_name
        normalized_path = '/'.join(path_parts)

    if PROJECTS_BASEPATH in secret_path:
        secret_name = '.'.join(normalized_path[len(PROJECTS_BASEPATH) + 1:].split('/'))
    else:
        secret_name = '.'.join(normalized_path.split('/'))

    client.connect(SECDIST_HOST)

    stdin, stdout, stderr = client.exec_command('cat %s' % secret_path)
    secret_content = stdout.read()
    vault_secret = VaultSecret(secret_name, secret_content)

    yield vault_secret

    client.close()


def print_store_message(what, where):
    print 'Going to store', what.name, 'to', where.client.host


def match_secret(expression, secret_to_find):
    if re.search(expression, secret_to_find):
        return True
    return False


def read_recurse_from_directory(directory_name, base_path=PROJECTS_BASEPATH, regex=None, excluded_dirs=[]):
    secrets_root_directory = os.path.join(base_path, directory_name)
    print "\nReading secrets from '%s'" % secrets_root_directory

    client.connect(SECDIST_HOST)
    stdin, stdout, stderr = client.exec_command('cd %s && ls' % secrets_root_directory)

    entity_list = [key_name for key_name in stdout.read().split('\n') if key_name]
    client.close()

    for entity in entity_list:
        client.connect(SECDIST_HOST)
        stdin, stdout, stderr = client.exec_command(
            'cd %s && python -c "import os; print os.path.isdir(\'%s\')"' % (secrets_root_directory, entity)
        )
        if stdout.read().strip() == "True":
            if entity in excluded_dirs:
                "%s is excluded, not storing."
            else:
                print entity, "is directory, searching for files with secrets..."
                read_recurse_from_directory(
                    os.path.join(secrets_root_directory, entity), regex=regex, excluded_dirs=excluded_dirs
                )

        else:
            if (regex is None) or (regex == ''):
                with read_secret(os.path.join(directory_name, entity), normalize=True) as vault_secret:
                    print_store_message(vault_secret, CRYPTA_VAULT_CLIENT)

                if not DRY_RUN:
                    CRYPTA_VAULT_CLIENT.store(vault_secret.name, vault_secret)
            else:
                if match_secret(regex, entity):
                    with read_secret(os.path.join(directory_name, entity), normalize=True) as vault_secret:
                        print_store_message(vault_secret, CRYPTA_VAULT_CLIENT)

                    if not DRY_RUN:
                        CRYPTA_VAULT_CLIENT.store(vault_secret.name, vault_secret)

        client.close()


if __name__ == "__main__":
    secrets_dir = args.secrets_dir
    DRY_RUN = args.dry_run
    PRODUCTION = args.is_production
    EXPRESSION = args.expression
    EXCLUDE_DIRS = args.exclude_dirs
    CRYPTA_VAULT_CLIENT.set_client_of_environment(PRODUCTION)

    print "Migrating secrets from project path \"%s\" from subdir \"%s\"" % (PROJECTS_BASEPATH, secrets_dir)
    if EXCLUDE_DIRS is not None:
        EXCLUDE_DIRS = EXCLUDE_DIRS.split(',')
        print "Dirs excluded:", ', '.join(EXCLUDE_DIRS)
    else:
        EXCLUDE_DIRS = []
        print '\n'

    read_recurse_from_directory(secrets_dir, base_path=PROJECTS_BASEPATH, regex=EXPRESSION, excluded_dirs=EXCLUDE_DIRS)
    print "\nAll done!"
