#include "reader.h"
#include <library/cpp/json/json_reader.h>
#include <library/cpp/string_utils/base64/base64.h>
#include <util/string/split.h>
#include <util/string/util.h>
#include <algorithm>

using NCrypta::NSoup::NBB::TLinks;

namespace {
    const TString DEFAULT_FILTER = ".*";

    TString extractDeviceInfo(const TString& browserInfo, const TString& stopKey) {
        TVector<TString> res;
        const TString *key;
        bool iskey = true;

        Split(browserInfo, ":", res);
        for (const auto& x : res) {
            if (iskey) {
                key = &x;
            } else {
                if (*key == stopKey) {
                    return std::move(x);
                }
            }
            iskey = !iskey;
        }

        return {};
    }

    NJson::TJsonValue ToJson(const TString& input) {
        NJson::TJsonValue result;
        NJson::ReadJsonTree(input, &result);

        return result;
    }

    TString MakeLogEntry(ui64 timestamp, const TString& yuid, const NJson::TJsonValue& json) {
        TStringBuilder builder;

        builder << TInstant::Now() << ": timestamp: " << timestamp << ", yandexuid: " << yuid;
        for (const auto& [key, value] : json.GetMap()) {
            builder << ", " << key << ": " << value;
        }

        return builder;
    }
}

TFilteredReader::TFilteredReader(const TFilteredReaderOptions& options)
    : Options(options)
    , Running(false)
    , TvmKeyManager(Options.GetTvmOptions(), {Options.GetLogbrokerOptions().GetServerTvmId()})
    , LogbrokerPullerHandle(CreateLogBrokerPuller(Options.GetLogbrokerOptions(), TvmKeyManager, {}))
    , DecryptorPtr(MakeHolder<NCrypta::NGraph::TDecryptor>(Options.GetMetrikaKey()))
    , SenderQueue(TThreadPool::TParams().SetBlocking(true).SetCatching(true))
{
    for (const auto& filter : Options.GetFilter()) {
        FilterExpressions.emplace_back(MakeHolder<re2::RE2>(filter));
    }

    if (FilterExpressions.empty()) {
        FilterExpressions.emplace_back(MakeHolder<re2::RE2>(DEFAULT_FILTER));
    }

    SenderQueue.Start(Options.GetWorkers(), 100 * Options.GetWorkers());
    Y_VERIFY(!Options.GetMetrikaKey().empty());
}

bool TFilteredReader::IsMatched(const TString& value) {
    return std::any_of(
        FilterExpressions.begin(),
        FilterExpressions.end(),
        [&value](auto&& filterHolder) {
            return RE2::FullMatch(value, *filterHolder);
        }
    );
}

void TFilteredReader::DecryptAndPrint(const TString data) {
    static const TString biKey = "browserinfo";
    static const TString tsKey = "timestamp";
    static const TString yuidKey = "yandexuid";

    try {
        auto logData = ToJson(data);
        auto deviceInfoEncrypted = extractDeviceInfo(logData[biKey].GetString(), "di");
        if (deviceInfoEncrypted.empty()) {
            return;
        }

        RemoveAll(deviceInfoEncrypted, '\n');
        const auto& decrypted = DecryptorPtr->Decrypt(Base64Decode(deviceInfoEncrypted));
        auto deviceInfo = ToJson(decrypted);

        const auto timestamp = logData[tsKey].GetUInteger();
        const auto& yandexuid = logData[yuidKey].GetString();
        if (IsMatched(yandexuid)) {
            Cout << MakeLogEntry(timestamp, yandexuid, deviceInfo) << Endl;
            return;
        }

        for (const auto& [key, value] : deviceInfo.GetMap()) {
            if (IsMatched(value.GetString())) {
                Cout << MakeLogEntry(timestamp, yandexuid, deviceInfo) << Endl;
                return;
            }
        }
    } catch(...) {
        return;
    }
}

void TFilteredReader::CallBack(const TStringBuf data) {
    auto unused = SenderQueue.AddFunc([this, data]() {
        DecryptAndPrint(TString{data});
    });

    Y_UNUSED(unused);
}

void TFilteredReader::Stop() {
    AtomicSwap(&Running, false);
}

void TFilteredReader::Start() {
    if (AtomicSwap(&Running, true)) {
        return;
    }

    auto needStop = [this]() {
        return !IsRunning();
    };
    auto callBack = [this](const TStringBuf data) {
        CallBack(data);
    };

    LogbrokerPullerHandle->Process(callBack, needStop);
}
