import argparse
import datetime
import logging
import os
import time

import yt.wrapper as yt

from crypta.lib.python import argparse_utils
from crypta.lib.python.logging import logging_helpers
import crypta.utils.upload_to_logbroker.lib as upload_to_logbroker


logger = logging.getLogger(__name__)


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument("--yt-proxy", required=True, help="YT proxy to use")
    parser.add_argument("--yt-pool", required=True)

    parser.add_argument("--source-dir", required=True, help="Tables to push to Logbroker")
    parser.add_argument("--value-column", required=True, help="Column with values for push-client")
    parser.add_argument("--drop-src", type=argparse_utils.boolean_as_string, required=True,
                        help="Remove tables after dumping them successfuly")

    parser.add_argument("--errors-dir", required=True, help="Push client errors dir as YT")
    parser.add_argument("--errors-ttl", required=True, help="TTL for push client errors, days", type=int)

    parser.add_argument("--push-client-file-path", default="/usr/bin/push-client", help="Path to push-client binary")

    parser.add_argument("--logbroker-hostname", required=True, help="LogBroker hostname")
    parser.add_argument("--logbroker-ident", required=True, help="LogBroker client for writing logs")
    parser.add_argument("--log-type", required=True, help="LogBroker target log type")

    parser.add_argument("--logbroker-master-port", default=None, help="LogBroker client for writing logs", type=int)
    parser.add_argument("--logbroker-data-port", default=None, help="LogBroker target log type", type=int)

    return parser.parse_args()


def main():
    logging_helpers.configure_stdout_logger(logging.getLogger())
    args = parse_args()
    yt.config.set_proxy(args.yt_proxy)
    yt.config["pool"] = args.yt_pool

    logger.info("Pushing tables to LogBroker")

    try:
        if not os.path.exists(args.push_client_file_path):
            raise RuntimeError("Path does not exist: {}".format(args.push_client_file_path))

        tables = yt.list(args.source_dir, absolute=True)
        if not tables:
            logger.info("No tables to push to LB. Exiting.")
            exit(0)

        upload_mapper = upload_to_logbroker.get_command(args.logbroker_hostname, args.logbroker_ident, args.log_type,
                                                        args.logbroker_master_port, args.logbroker_data_port)
        if not yt.exists(args.errors_dir):
            yt.mkdir(args.errors_dir, recursive=True)

        timestamp = str(int(time.time()))
        push_client_output_table = yt.ypath_join(args.errors_dir, timestamp)
        errors_ttl = datetime.timedelta(days=args.errors_ttl)

        upload_to_logbroker.upload_to_logbroker(tables, push_client_output_table, args.push_client_file_path,
                                                args.value_column, args.drop_src, upload_mapper, errors_ttl)
    except Exception:
        logger.exception("Failed to push tables to LogBroker")
        exit(1)

    logger.info("Finished successfully")
