import logging

import yt.wrapper as yt

from crypta.lib.python.yt import yt_helpers
from crypta.utils.upload_to_logbroker.lib import mapper


logger = logging.getLogger(__name__)


def get_command(hostname, ident, log_type, master_port=None, data_port=None, additional_params=None):
    params = {
        "network:master-addr": hostname,
        "network:transport": "ipv6",
        "ident": ident,
        "log-type": log_type,
        "network:master-port": master_port,
        "network:data-port": data_port
    }

    if additional_params:
        params.update(additional_params)

    args = " ".join("--{}={}".format(key, value) for key, value in params.iteritems() if value)
    return './push-client --stdin {} 2>&1 | xargs'.format(args)


def upload_to_logbroker(src_tables, push_client_output_table, push_client_file_path, column, drop_src,
                        upload_mapper, error_ttl):
    has_errors = False

    with yt.Transaction(), yt.TempTable() as raw_push_client_output_table:
        logger.info("Pushing tables %s to LB, raw output at %s", src_tables, raw_push_client_output_table)
        yt.run_map(upload_mapper, src_tables, raw_push_client_output_table, local_files=[push_client_file_path],
                   format=yt.SchemafulDsvFormat(columns=[column], enable_escaping=False))

        logger.info("Extracting errors from push-client output")
        yt.run_map(mapper.ParsePushClientOutputMapper(column), raw_push_client_output_table, push_client_output_table)

        if yt.get(yt.ypath_join(push_client_output_table, "@row_count")):
            logger.info("Not dropping source tables, because push-client had errors.")
            has_errors = True
        elif drop_src:
            logger.info("Dropping source tables %s", src_tables)
            for table in src_tables:
                yt.remove(table)
        else:
            logger.info("Not dropping source tables")

    yt_helpers.set_ttl(push_client_output_table, error_ttl)
    if has_errors:
        raise Exception("Push client had errors.")

    logger.info("Successfuly exported %s to LogBroker", src_tables)
