import os

import argparse
import logging
import logging.config
import requests
import socket
import sys
import yaml

from crypta.lib.python import cmd


YAV_KEY_FILE_NAME = 'yav_key'

logger = logging.getLogger(__name__)


def main():
    parser = argparse.ArgumentParser(description='Crypta vault client')
    parser.add_argument('--config', help='Config file', required=True)
    parser.add_argument('--log-config', help='Logger config file', required=True)

    args = parser.parse_args()

    with open(args.config) as f:
        config = yaml.safe_load(f)

    with open(args.log_config) as f:
        log_config = yaml.safe_load(f)
        logging.config.dictConfig(log_config)

    logger.info('=== SECRETS CONFIGURATION STARTED ===')

    try:
        yav_key_file = os.path.join(config['secrets-dir'], YAV_KEY_FILE_NAME)
        if not os.path.isfile(yav_key_file):
            logger.warn('No yav key file found')

            if not os.path.isdir(config['secrets-dir']):
                logger.info('Creating secrets dir: %s', config['secrets-dir'])
                os.makedirs(config['secrets-dir'], 0755)
            else:
                logger.info('Found secrets dir: %s', config['secrets-dir'])

            cmd.run(['setfacl', '-m', 'u:{user}:rwx'.format(user=config['vault-user']), config['secrets-dir']], description='Set ACL on secrets dir')

            logger.info('Requesting yav key file from vault')
            vault_response = requests.get('{vault}?host={hostname}'.format(vault=config['vault'], hostname=socket.getfqdn()))
            if vault_response.status_code != requests.codes.ok:
                raise Exception('Failed to get yav key from vault. Server responded with error: %s', vault_response.text)

            logger.info('Changing mode on yav key file')
            os.chmod(yav_key_file, 0600)
        else:
            logger.info('Found yav key file %s', yav_key_file)

        cmd.run(
            ['yav-deploy', '--rsa-login', config['yav-key-user'], '--rsa-private-key', yav_key_file, '--debug'],
            attempts=3,
            timeout_seconds=30,
            description='Run yav-deploy'
        )

        logger.info('=== SECRETS CONFIGURATION SUCCEEDED ===')
    except Exception:
        logger.exception('Exception')
        logger.error('=== SECRETS CONFIGURATION FAILED ===')
        sys.exit(1)
