import os

import argparse
import flask
import logging
import logging.config
from gevent import pywsgi
import yaml

from crypta.lib.python import cmd
from crypta.utils.vault.server import lib


REMOTE_USER_ENV = 'REMOTE_USER'
REMOTE_USER_PRIVATE_KEY_ENV = 'REMOTE_USER_PRIVATE_KEY'
SECRET_ENV = 'SECRET'

REMOTE_USER_PRIVATE_KEY_FILE = 'private_key'
SECRET_FILE = 'secret'
SECRET_REMOTE_PATH = '/etc/crypta/secrets/yav_key'


app = flask.Flask(__name__)


class HttpError(Exception):
    def __init__(self, status_code, message):
        Exception.__init__(self)
        self.message = message
        self.status_code = status_code


@app.route("/ping")
def ping():
    return "OK"


@app.route('/')
def root():
    host = flask.request.args.get('host')
    logging.info('Received host: {}'.format(host))

    if not host:
        raise HttpError(400, 'No host provided')

    if not lib.is_allowed_host(host):
        raise HttpError(403, 'Host is not allowed: {}'.format(host))

    cmd.run([
        'scp',
        '-o', 'StrictHostKeyChecking=no',
        '-o', 'UserKnownHostsFile=/dev/null',
        '-i', REMOTE_USER_PRIVATE_KEY_FILE,
        SECRET_FILE,
        '{user}@{host}:{path}'.format(user=os.environ[REMOTE_USER_ENV], host=host, path=SECRET_REMOTE_PATH)
    ], description='Copy secret to remote host')

    return 'OK'


@app.errorhandler(Exception)
def on_error(e):
    if not isinstance(e, HttpError):
        logging.exception(e)

    return flask.make_response(e.message, e.status_code if hasattr(e, 'status_code') else 500)


def main():
    parser = argparse.ArgumentParser(description='Crypta vault server')
    parser.add_argument('--host', help='Host', default="0.0.0.0")
    parser.add_argument('--port', help='Port', required=True, type=int)
    parser.add_argument('--log-config', help='Logger config file', required=True)

    args = parser.parse_args()

    with open(args.log_config) as f:
        config = yaml.safe_load(f)
        logging.basicConfig(**config)

    logging.info('--- START ---')

    with open(SECRET_FILE, 'w') as f:
        f.write(os.environ[SECRET_ENV])

    with open(REMOTE_USER_PRIVATE_KEY_FILE, 'w') as f:
        f.write(os.environ[REMOTE_USER_PRIVATE_KEY_ENV])

    os.chmod(REMOTE_USER_PRIVATE_KEY_FILE, 0700)

    server = pywsgi.WSGIServer((args.host, args.port), app)
    server.serve_forever()
