import concurrent.futures
import datetime
import logging

from crypta.lib.python.getoptpb import ParsePbOptions
from crypta.lib.python.logging import logging_helpers
from crypta.lib.python.yt import yt_helpers
from crypta.utils.yt_compress_tables.config_pb2 import TConfig

logger = logging.getLogger(__name__)


def main():
    logging_helpers.configure_stdout_logger(logger)

    config = ParsePbOptions(TConfig)
    yt_client = yt_helpers.get_yt_client(config.Yt.Proxy, config.Yt.Pool)
    settings = yt_client.get_attribute(config.Directory, "nightly_compression_settings")
    logger.info("Settings for %s: %s", config.Directory, settings)

    age_threshold = (datetime.datetime.now() - datetime.timedelta(seconds=settings.get("min_table_age", 0))).isoformat()
    min_size_bytes = settings.get("min_table_size", 0)
    pool = settings.get("pool", "crypta_all")
    compression_codec = settings.get("compression_codec", "zlib_9")
    erasure_codec = settings.get("erasure_codec", "lrc_12_2_2")

    tables = list(yt_client.search(
        config.Directory,
        "table",
        attributes=("resource_usage", "creation_time", "erasure_codec", "compression_codec"),
        object_filter=lambda x:
            x.attributes["resource_usage"]["disk_space"] >= min_size_bytes and
            x.attributes["creation_time"] <= age_threshold and
            (
                x.attributes["erasure_codec"] != erasure_codec or
                x.attributes["compression_codec"] != compression_codec
            )
    ))

    logger.info("Table list: %s", [str(x) for x in tables])

    def compress(table):
        logger.info("Compressing: %s", table)
        client = yt_helpers.get_yt_client(config.Yt.Proxy, pool)
        with client.Transaction():
            client.transform(str(table), erasure_codec=erasure_codec, compression_codec=compression_codec)
        logger.info("Done: %s", table)

    with concurrent.futures.ThreadPoolExecutor(config.Workers) as executor:
        list(executor.map(compress, tables))

    logger.info("Done")
