"""
Gets pools statistics from YT
"""

import argparse
import logging
import logging.config

from crypta.lib.python import yaml_config
from crypta.lib.python.graphite import sender
from crypta.lib.python.yt import yt_helpers
from crypta.utils.yt_stats.lib import (
    helpers,
    ops
)


logger = logging.getLogger(__name__)


def parse_args(description=None):
    parser = argparse.ArgumentParser(description=description)
    parser.add_argument("--config", required=True, help="Config file")
    parser.add_argument('--log-config', required=True, help='Logger config file')
    return parser.parse_args()


def main():
    args = parse_args(description=__doc__)
    config = yaml_config.load(args.config)
    logging.config.fileConfig(args.log_config, disable_existing_loggers=False)

    logger.info(args)
    logger.info(config)

    graphite_sender = sender.GraphiteSender(fqdn=config["fake_hostname_for_graphite"], dry_run=not config["send_data_to_graphite"])

    yt_client = yt_helpers.get_yt_client(config["yt_proxy"], yt_pool=config["yt_pool"])
    with yt_client.Transaction():
        input_table = helpers.get_input_table(yt_client, config["event_log_table"], config["row_range"])
        output_table = yt_client.create_temp_table(path=config["yt_tmp_dir"], prefix="yt-stats-")
        mapper = ops.PoolStatisticsMapper(config["time_range_hours"])
        reducer = ops.PoolStatisticsReducer(config["pool_regexp"])
        logger.info("Time range %s - %s UTC", mapper.timestamp_lower_bound, mapper.timestamp_upper_bound)
        yt_client.run_map_reduce(mapper, reducer, input_table, output_table, reduce_by=["tree_snapshot_id"])

        for row in yt_client.read_table(output_table):
            metrics = helpers.get_metrics_from_row(row)
            graphite_sender.send_metrics(metrics)

        yt_client.remove(output_table)
