import calendar
import collections
import datetime
import os
import re

from crypta.lib.python import time_utils


EVENT_TYPE_FIELD = "event_type"
OPERATIONS_FIELD = "operations"
TIMESTAMP_FIELD = "timestamp"

DATETIME_FORMAT = "%Y-%m-%dT%H:%M:%S.%fZ"


def get_utcnow():
    var = time_utils.CRYPTA_FROZEN_TIME_ENV
    if var in os.environ:
        ts = int(os.environ[var])
        return datetime.datetime.utcfromtimestamp(ts)
    return datetime.datetime.utcnow()


class PoolStatisticsMapper(object):
    def __init__(self, time_range_hours):
        self.timestamp_upper_bound = get_utcnow()  # NOTE: scheduler event log stores time in UTC
        self.timestamp_lower_bound = self.timestamp_upper_bound - datetime.timedelta(hours=time_range_hours)

    def __call__(self, row):
        timestamp = datetime.datetime.strptime(row[TIMESTAMP_FIELD], DATETIME_FORMAT)
        if (row[EVENT_TYPE_FIELD] == "fair_share_info") and (self.timestamp_lower_bound < timestamp < self.timestamp_upper_bound) and (OPERATIONS_FIELD in row):
            yield row


class PoolStatisticsReducer(object):
    def __init__(self, pool_regexp):
        self.pool_regexp = re.compile(pool_regexp)

    def __call__(self, key, rows):
        pools = collections.defaultdict(lambda: {
            "operations_counts": 0,
            "usage_ratio": 0,
            "demand_ratio": 0
        })

        for row in rows:
            for operation in row[OPERATIONS_FIELD].values():
                pool = operation["pool"]
                if not self.pool_regexp.match(pool):
                    continue

                pools[pool]["operations_counts"] += 1
                pools[pool]["usage_ratio"] += operation["usage_ratio"]
                pools[pool]["demand_ratio"] += operation["demand_ratio"]

        timestamp = datetime.datetime.strptime(row[TIMESTAMP_FIELD], DATETIME_FORMAT)

        for pool, statistics in pools.iteritems():
            yield {
                "pool": pool,
                "operations_counts": statistics["operations_counts"],
                "usage_ratio": statistics["usage_ratio"],
                "demand_ratio": statistics["demand_ratio"],
                "timestamp": calendar.timegm(timestamp.utctimetuple())
            }
