"use strict";

const path = require("path");
const webpack = require("webpack");
const HtmlWebpackPlugin = require("html-webpack-plugin");
const CopyWebpackPlugin = require("copy-webpack-plugin");
const CaseSensitivePathsPlugin = require("case-sensitive-paths-webpack-plugin");
const MiniCssExtractPlugin = require("mini-css-extract-plugin");

const CWD = process.cwd();

const indexHtmlPath = path.resolve(CWD, "public/index.html");
const envJsonPath = path.resolve(CWD, "public/env.json");
const faviconPath = path.resolve(CWD, "public/favicon.ico");
const modulesPath = path.join(CWD, "node_modules");
const buildPath = path.join(CWD, "build");
const srcPath = path.join(CWD, "src");
const libPath = path.join(CWD, "..", "lib");

const isProduction = process.env.NODE_ENV === "production";
const mode = isProduction ? "production" : "development";
const devtool = isProduction ? "hidden-source-map" : "source-map";

class CleanUpStatsPlugin {
    shouldPickStatChild(child) {
        return child.name.indexOf("mini-css-extract-plugin") !== 0;
    }

    apply(compiler) {
        compiler.hooks.done.tap("CleanUpStatsPlugin", (stats) => {
            const children = stats.compilation.children;
            if (Array.isArray(children)) {
                stats.compilation.children = children.filter((child) =>
                    this.shouldPickStatChild(child)
                );
            }
        });
    }
}

let plugins = [];
if (isProduction) {
    plugins = [
        new HtmlWebpackPlugin({
            inject: true,
            template: indexHtmlPath,
            favicon: faviconPath,
            minify: {
                removeComments: true,
                collapseWhitespace: true,
                removeRedundantAttributes: true,
                useShortDoctype: true,
                removeEmptyAttributes: true,
                removeStyleLinkTypeAttributes: true,
                keepClosingSlash: true,
                minifyJS: true,
                minifyCSS: true,
                minifyURLs: true,
            },
        }),
        new webpack.DefinePlugin({
            "process.env.NODE_ENV": '"production"',
            IS_DEVELOP: false,
        }),
        new MiniCssExtractPlugin({
            filename: "static/css/[name].[hash:8].css",
            chunkFilename: "static/css/[id].[hash:8].css",
        }),
    ];
} else {
    plugins = [
        new HtmlWebpackPlugin({
            inject: true,
            template: indexHtmlPath,
            favicon: faviconPath,
        }),
        new CopyWebpackPlugin([{ from: envJsonPath, to: buildPath }]),
        new webpack.DefinePlugin({
            "process.env.NODE_ENV": '"development"',
            IS_DEVELOP: true,
        }),
        new MiniCssExtractPlugin({
            filename: "static/css/[name].css",
            chunkFilename: "static/css/[id].css",
        }),
        new webpack.HotModuleReplacementPlugin(),
        new CaseSensitivePathsPlugin(),
        new CleanUpStatsPlugin(),
    ];
}

const config = {
    mode: mode,
    bail: true,
    devtool: devtool,
    entry: [path.join(srcPath, "index")],
    output: {
        path: buildPath,
        pathinfo: false,
        publicPath: "/",
        filename: "static/js/bundle.[hash:8].js",
        chunkFilename: "static/js/[name].[chunkhash:8].chunk.js",
    },
    resolve: {
        modules: [modulesPath],
        alias: {
            styles: path.resolve(CWD, "src/styles"),
            "@lib": libPath,
        },
    },
    module: {
        rules: [
            {
                test: /\.jsx?$/,
                include: [srcPath, libPath],
                exclude: /node_modules/,
                loader: "babel-loader",
            },
            {
                test: /\.s?css$/,
                use: [
                    {
                        loader: MiniCssExtractPlugin.loader,
                        options: {
                            hmr: !isProduction,
                        },
                    },
                    "css-loader",
                    "sass-loader",
                ],
            },
            {
                test: /\.(ico|jpg|png|gif)$/,
                use: [
                    {
                        loader: "file-loader",
                        options: {
                            name: "[name].[hash:8].[ext]",
                            outputPath: "static/media",
                        },
                    },
                ],
            },
            {
                test: /\.(svg|eot|ttf|otf|woff|woff2)$/,
                loader: "url-loader",
            },
        ],
    },
    plugins: plugins,
};

module.exports = config;
