import os

import sqlalchemy.orm
import sqlalchemy.engine.url
import sqlalchemy.pool
import sqlalchemy.ext.declarative


@sqlalchemy.ext.declarative.as_declarative()
class Base(object):
    """
    Add some default properties and methods to the SQLAlchemy declarative base.
    """

    @property
    def columns(self):
        return [c.name for c in self.__table__.columns]

    @property
    def columnitems(self):
        return dict([(c, getattr(self, c)) for c in self.columns])

    def __repr__(self):
        return '{}({})'.format(self.__class__.__name__, self.columnitems)

    def tojson(self):
        return self.columnitems


def create_url(**kwargs):
    return sqlalchemy.engine.url.URL(**kwargs)


def _is_sqlite(url):
    if isinstance(url, sqlalchemy.engine.url.URL):
        return url.drivername == 'sqlite'
    elif isinstance(url, str):
        return url.startswith('sqlite')
    else:
        raise ValueError('Invalid database url: ' + str(url))


def create_engine(url):
    engine_args = {}
    # check_same_thread + StaticPool is a workaround to make it
    # work with sqlite inmemory database (for development purposes)
    if _is_sqlite(url):
        engine_args.update(
            poolclass=sqlalchemy.pool.StaticPool,
            connect_args={'check_same_thread': False}
        )

    engine = sqlalchemy.create_engine(url, **engine_args)
    Base.metadata.create_all(engine)
    return engine


def create_session(engine):
    return sqlalchemy.orm.scoped_session(sqlalchemy.orm.sessionmaker(bind=engine))


def get_session(db):
    database = {
        'database': 'dating',
        'drivername': 'postgresql',
        'host': 'special.database.crypta.yandex.net',
        'password': os.environ.get('DB_PASS'),
        'port': 6432,
        'username': 'crypta'
    }

    url = create_url(**database)
    engine = create_engine(url)
    session = create_session(engine)
    return session, engine
