import React from "react";
import { connect } from "react-redux";
import PropTypes from "prop-types";

import { Grid, Row, Col } from "react-bootstrap";
import { Button } from "@yandex-lego/components/Button/desktop/bundle";
import { Spin } from "@yandex-lego/components/Spin/desktop/bundle";

import CampaignsPanel from "./CampaignsPanel";
import AdGroupsPanel from "./AdGroupsPanel";
import BannersPanel from "./BannersPanel";
import PhrasesPanel from "./PhrasesPanel";
import ModalContainer from "../ModalContainer";

import { initializePreviewPanelIfNeeded } from "../actions/Preview";
import { upload, hideUploadingInfo } from "../actions/Upload";
import { backOnUploadingPanel } from "../actions";

import "./index.scss";

const ButtonsEnum = Object.freeze({
    UPLOAD: "Загрузить кампанию",
    BACK: "Назад",
});

class PreviewPanel extends React.Component {
    componentDidMount() {
        const { templateId, onInitializePreviewPanel } = this.props;
        onInitializePreviewPanel(templateId);
    }

    getFormattedUploadingInfo(client) {
        return (
            <div>
                Имя кампании: {client.campaign.name}
                <br />
                Клиент: {client.name}
                <br />
                Статус: {client.uploadingStatus && client.uploadingStatus.message}
            </div>
        );
    }

    renderPreview() {
        const { templateId, client, campaignId, clientId, api, onUpload, onBack, onHideUploadingInfo } = this.props;
        return (
            <div>
                <Grid fluid={true}>
                    <Row>
                        <Col md={12}>
                            <div className="FieldLayout">
                                <span className="pull-right">
                                    <Button view="default" theme="raised" size="s" onClick={() => onUpload(client)}>
                                        {ButtonsEnum.UPLOAD}
                                    </Button>
                                    <Button view="default" theme="raised" size="s" onClick={() => onBack()}>
                                        {ButtonsEnum.BACK}
                                    </Button>
                                </span>
                            </div>
                            <ModalContainer
                                visible={client.uploadingStatus && client.uploadingStatus.visible}
                                onClose={() => onHideUploadingInfo(clientId)}
                            >
                                {this.getFormattedUploadingInfo(client)}
                            </ModalContainer>
                        </Col>
                    </Row>
                    <Row>
                        <Col md={12}>
                            <Row>
                                <Col md={4}>
                                    <div className="FieldLayout">
                                        <CampaignsPanel clientId={clientId} templateId={templateId} api={api} />
                                    </div>
                                    <div className="FieldLayout">
                                        <AdGroupsPanel
                                            campaignId={campaignId}
                                            clientId={clientId}
                                            templateId={templateId}
                                            api={api}
                                        />
                                    </div>
                                </Col>
                                <Col md={4}>
                                    <BannersPanel clientId={clientId} />
                                </Col>
                                <Col md={4}>
                                    <div className="FieldLayout">
                                        <PhrasesPanel />
                                    </div>
                                </Col>
                            </Row>
                        </Col>
                    </Row>
                </Grid>
            </div>
        );
    }

    renderSpin() {
        return <Spin progress={true} size="l" />;
    }

    render() {
        const { isLoaded } = this.props;
        if (isLoaded) {
            return <div>{this.renderPreview()}</div>;
        } else {
            return <div className="loading">{this.renderSpin()}</div>;
        }
    }
}

function getTemplateId(state, clientId) {
    const clientIndex = state.clients.findIndex((value) => value.id === clientId);
    return state.clients[clientIndex].template.id;
}

function getClient(state, clientId) {
    const clientIndex = state.clients.findIndex((value) => value.id === clientId);
    return state.clients[clientIndex];
}

function mapStateToProps(state, ownProps) {
    return {
        isLoaded: state.previewPanel.isLoaded,
        campaignId: state.previewPanel.campaigns.active,
        templateId: getTemplateId(state, state.activeClientId),
        adGroupId: state.previewPanel.adGroups.active,
        client: getClient(state, ownProps.clientId),
    };
}

function mapDispatchToProps(dispatch, ownProps) {
    return {
        onInitializePreviewPanel: (templateId) =>
            dispatch(initializePreviewPanelIfNeeded(ownProps.api, ownProps.clientId, templateId)),
        onUpload: (client) => dispatch(upload(ownProps.api, client)),
        onBack: () => dispatch(backOnUploadingPanel()),
        onHideUploadingInfo: (clientId) => dispatch(hideUploadingInfo(clientId)),
    };
}

PreviewPanel.propTypes = {
    api: PropTypes.object.isRequired,
    clientId: PropTypes.number.isRequired,
};

export default connect(mapStateToProps, mapDispatchToProps)(PreviewPanel);
