import React from "react";
import { connect } from "react-redux";
import PropTypes from "prop-types";
import FontAwesome from "react-fontawesome";
import Dropzone from "react-dropzone";

import ModalContainer from "../ModalContainer";

import { uploadImages, showImageInfo, hideImageInfo } from "../actions/Upload";
import { APIStatusEnum } from "../actions/utils";

import "./ImagesPanel.scss";

class DropzoneContainer extends React.Component {
    render() {
        return (
            <div>
                <Dropzone className="DropzoneBlock" onDrop={this.props.onLoad}>
                    <div className="DropzoneText">
                        <div>
                            <FontAwesome name="download-alt" />
                        </div>
                        <div>Перетащи картинки</div>
                    </div>
                </Dropzone>
            </div>
        );
    }
}

class ImagesPanel extends React.Component {
    renderPreview(image) {
        const { onShowImageInfo } = this.props;

        let imageClassName;

        switch (image.uploadingStatus.status) {
            case APIStatusEnum.SUCCESS:
                imageClassName = "img-thumbnail ImagePreview ImageStatusSuccess";
                break;
            case APIStatusEnum.FAILED:
                imageClassName = "img-thumbnail ImagePreview ImageStatusFailed";
                break;
            default:
                imageClassName = "img-thumbnail ImagePreview ImageStatusLoading";
        }

        const imageSrc = image.direct.url || image.file.preview;

        return (
            <img
                key={image.id}
                className={imageClassName}
                alt={"?"}
                src={imageSrc}
                onClick={() => onShowImageInfo(image.id)}
            />
        );
    }

    renderPreviews() {
        const { images } = this.props;
        if (images.length > 0) {
            const imagePreviews = images.map((image) => this.renderPreview(image));
            return (
                <div className="FieldLayout">
                    <div>{imagePreviews}</div>
                </div>
            );
        } else {
            return null;
        }
    }

    renderImageInfo() {
        const { infoViewImage } = this.props;
        if (infoViewImage !== null) {
            return (
                <div>
                    Статус: {infoViewImage.uploadingStatus.message}
                    <br />
                    Размеры: {String(infoViewImage.size.width) + " x " + String(infoViewImage.size.height)}
                    <br />
                    Хэш:{" "}
                    {infoViewImage.uploadingStatus.status === APIStatusEnum.SUCCESS ? infoViewImage.direct.hash : "-"}
                    <br />
                </div>
            );
        } else {
            return null;
        }
    }

    render() {
        const { onImagesLoad, infoViewImage, onHideImageInfo } = this.props;
        return (
            <div>
                <div className="FieldLayout">
                    <div>Картинки</div>
                    <DropzoneContainer onLoad={(images) => onImagesLoad(images)} />
                </div>
                {this.renderPreviews()}

                {/*Warning: Don't move ModalContainer on deeper layer. Do not know why, but it does not work*/}
                <ModalContainer visible={infoViewImage !== null} onClose={() => onHideImageInfo()}>
                    {this.renderImageInfo()}
                </ModalContainer>
            </div>
        );
    }
}

function getImages(state, clientId) {
    const clientIndex = state.clients.findIndex((value) => value.id === clientId);
    return state.clients[clientIndex].images;
}

function getInfoViewImage(state, clientId) {
    const clientIndex = state.clients.findIndex((value) => value.id === clientId);
    const client = state.clients[clientIndex];
    if (client.infoViewImageId === null) {
        return null;
    } else {
        const infoViewImageIndex = client.images.findIndex((image) => image.id === client.infoViewImageId);
        return state.clients[clientIndex].images[infoViewImageIndex];
    }
}

function mapStateToProps(state, ownProps) {
    return {
        images: getImages(state, ownProps.clientId),
        infoViewImage: getInfoViewImage(state, ownProps.clientId),
    };
}

function mapDispatchToProps(dispatch, ownProps) {
    return {
        onImagesLoad: (images) => dispatch(uploadImages(ownProps.api, ownProps.clientId, images)),
        onShowImageInfo: (imageId) => dispatch(showImageInfo(ownProps.clientId, imageId)),
        onHideImageInfo: () => dispatch(hideImageInfo(ownProps.clientId)),
    };
}

ImagesPanel.propTypes = {
    clientId: PropTypes.number.isRequired,
    api: PropTypes.object.isRequired,
};

export default connect(mapStateToProps, mapDispatchToProps)(ImagesPanel);
