import React from "react";
import { connect } from "react-redux";
import PropTypes from "prop-types";

import { RadioButton } from "@yandex-lego/components/RadioButton/desktop/bundle";
import { Checkbox } from "@yandex-lego/components/Checkbox/desktop/bundle";
import { isEqual } from "lodash";
import "./index.scss";
import "./RegionsPanel.scss";

import { selectCountry, selectExpandType, selectRegions } from "../actions/Upload";

class RegionsPanel extends React.Component {
    renderRegion(region, selectedRegionIds) {
        const { onRegionsSelect } = this.props;
        const checked = selectedRegionIds.findIndex((value) => value === region.direct_id) !== -1;
        return (
            <div key={region.direct_id} className="Checkbox NoMargin">
                <Checkbox
                    label={region.name}
                    size="s"
                    theme="normal"
                    checked={checked}
                    onChange={() => {
                        if (checked) {
                            onRegionsSelect(selectedRegionIds.filter((value) => value !== region.direct_id));
                        } else {
                            onRegionsSelect(selectedRegionIds.concat([region.direct_id]));
                        }
                    }}
                />
            </div>
        );
    }

    renderAllRegions(regionIds, selectedRegionIds) {
        const { onRegionsSelect } = this.props;
        const checked = isEqual(selectedRegionIds, regionIds);
        return (
            <div key="all" className="Checkbox NoMargin">
                <Checkbox
                    label={"Все"}
                    size="s"
                    theme="normal"
                    checked={checked}
                    onChange={() => {
                        if (checked) {
                            onRegionsSelect([]);
                        } else {
                            onRegionsSelect(regionIds);
                        }
                    }}
                />
            </div>
        );
    }

    render() {
        const { geo, country, countries, onRegionsSelect, onExpandTypeSelect, onCountrySelect } = this.props;

        if (geo !== null) {
            const regionIds = country.regions.map((region) => region.direct_id).sort();
            const selectedRegionIds = geo.regionIds.sort();
            const renderedRegions = country.regions
                .sort((first, second) => first.name.localeCompare(second.name))
                .map((region) => this.renderRegion(region, selectedRegionIds));

            return (
                <div>
                    <div className="FieldLayout">
                        <div>Страна</div>
                        <div>
                            <RadioButton
                                size="s"
                                view="default"
                                value={geo.countryName}
                                onChange={(event) => onCountrySelect(event.target.value)}
                                options={countries.map((country) => ({
                                    value: country.name,
                                    children: country.name,
                                }))}
                            />
                        </div>
                    </div>
                    <div className="FieldLayout">
                        <div>Тип разреза</div>
                        <div>
                            <RadioButton
                                size="s"
                                view="default"
                                value={geo.expandTypeName}
                                onChange={(event) => onExpandTypeSelect(event.target.value)}
                                options={country.expand_types.map((expandType) => ({
                                    value: expandType.name,
                                    children: expandType.name,
                                    disabled: !expandType.available,
                                }))}
                            />
                        </div>
                    </div>
                    <div className="FieldLayout">
                        <div>Из регионов</div>
                        <div className="RegionsBlock">
                            <div className="FieldLayout">
                                {this.renderAllRegions(regionIds, selectedRegionIds)}
                                {renderedRegions}
                            </div>
                        </div>
                    </div>
                </div>
            );
        } else {
            return null;
        }
    }
}

function getGeoClientId(state, clientId) {
    const clientIndex = state.clients.findIndex((value) => value.id === clientId);
    if (clientIndex === -1) {
        return null;
    } else {
        return state.clients[clientIndex].geo;
    }
}

function getCountryByGeo(state, geo) {
    if (geo === null) {
        return null;
    }
    const countryIndex = state.countries.findIndex((value) => value.name === geo.countryName);
    if (countryIndex === -1) {
        return null;
    } else {
        return state.countries[countryIndex];
    }
}

function mapStateToProps(state, ownProps) {
    const geo = getGeoClientId(state, ownProps.clientId);
    return {
        geo: geo,
        country: getCountryByGeo(state, geo),
        countries: state.countries,
    };
}

function mapDispatchToProps(dispatch, ownProps) {
    return {
        onCountrySelect: (countryName) => dispatch(selectCountry(ownProps.clientId, countryName)),
        onExpandTypeSelect: (expandTypeName) => dispatch(selectExpandType(ownProps.clientId, expandTypeName)),
        onRegionsSelect: (regionIds) => dispatch(selectRegions(ownProps.clientId, regionIds)),
    };
}

RegionsPanel.propTypes = {
    clientId: PropTypes.number.isRequired,
};

export default connect(mapStateToProps, mapDispatchToProps)(RegionsPanel);
