import { handleError } from "./utils";

// API

export const REQUEST_TEMPLATES = "REQUEST_TEMPLATES";
export const RECEIVE_TEMPLATES_FAILED = "RECEIVE_TEMPLATES_FAILED";
export const RECEIVE_TEMPLATES_SUCCESS = "RECEIVE_TEMPLATES_SUCCESS";

export const REQUEST_TEMPLATE = "REQUEST_TEMPLATE";
export const RECIEVE_TEMPLATE_SUCCESS = "RECIEVE_TEMPLATE_SUCCESS";
export const RECIEVE_TEMPLATE_FAILED = "RECIEVE_TEMPLATE_FAILED";

export const REQUEST_BIDDING = "REQUEST_BIDDING";
export const RECEIVE_BIDDING_SUCCESS = "RECEIVE_BIDDING_SUCCESS";
export const RECEIVE_BIDDING_FAILED = "RECEIVE_BIDDING_FAILED";

export const REQUEST_MODERATION = "REQUEST_MODERATION";
export const RECEIVE_MODERATION_SUCCESS = "RECEIVE_MODERATION_SUCCESS";
export const RECEIVE_MODERATION_FAILED = "RECEIVE_MODERATION_FAILED";

// View

export const LoadingStatus = Object.freeze({
    DRAFT: 0,
    LOADING: 1,
    SUCCESS: 2,
    FAILED: 3,
});

export { fetchTemplates, loadTemplate, moderateTemplate, bidTemplate };
import { getApiPendingStatus, getApiFailedStatus, getApiSuccessStatus } from "./utils";

// == Templates ==

function requestTemplates(clientId) {
    return {
        type: REQUEST_TEMPLATES,
        clientId,
        templatesStatus: getApiPendingStatus(),
    };
}

function receiveTemplatesSuccess(clientId, templates) {
    return {
        type: RECEIVE_TEMPLATES_SUCCESS,
        clientId,
        templates: templates,
        templatesStatus: getApiSuccessStatus(),
    };
}

function receiveTemplatesFailed(clientId, error) {
    return {
        type: RECEIVE_TEMPLATES_FAILED,
        clientId,
        templatesStatus: getApiFailedStatus(error),
    };
}

function fetchTemplates(api, clientId) {
    return (dispatch) => {
        dispatch(requestTemplates(clientId));
        return api
            .getTemplates(clientId)
            .then((templates) => {
                dispatch(receiveTemplatesSuccess(clientId, templates));
            })
            .catch((error) => {
                dispatch(receiveTemplatesFailed(clientId, error));
                throw error;
            });
    };
}

// == Moderation ==

function requestModerateTemplate(clientId, templateId) {
    return {
        type: REQUEST_MODERATION,
        clientId,
        templateId,
        moderationStatus: getApiPendingStatus(),
    };
}

function receiveModerateTemplateSuccess(clientId, templateId) {
    return {
        type: RECEIVE_MODERATION_SUCCESS,
        clientId,
        templateId,
        moderationStatus: getApiSuccessStatus(),
    };
}

function receiveModerateTemplateFailed(clientId, templateId, error) {
    return {
        type: RECEIVE_MODERATION_FAILED,
        clientId,
        templateId,
        moderationStatus: getApiFailedStatus(error),
    };
}

function moderateTemplate(api, clientId, templateId) {
    return (dispatch) => {
        dispatch(requestModerateTemplate(clientId, templateId));
        return api
            .moderateTemplate(clientId, templateId)
            .then(() => {
                dispatch(receiveModerateTemplateSuccess(clientId, templateId));
            })
            .catch((error) => {
                dispatch(receiveModerateTemplateFailed(clientId, templateId, error));
                throw error;
            });
    };
}

// == Bid ==

function requestBidTemplate(clientId, templateId) {
    return {
        type: REQUEST_BIDDING,
        clientId,
        templateId,
        biddingStatus: getApiPendingStatus(),
    };
}

function receiveBidTemplateSuccess(clientId, templateId) {
    return {
        type: RECEIVE_BIDDING_SUCCESS,
        clientId,
        templateId,
        biddingStatus: getApiSuccessStatus(),
    };
}

function receiveBidTemplateFailed(clientId, templateId, error) {
    return {
        type: RECEIVE_BIDDING_FAILED,
        clientId,
        templateId,
        biddingStatus: getApiFailedStatus(error),
    };
}

function bidTemplate(api, clientId, templateId, bid) {
    return (dispatch) => {
        dispatch(requestBidTemplate(clientId, templateId));
        return api
            .bidTemplate(clientId, templateId, { bid_value: bid })
            .then(() => {
                dispatch(receiveBidTemplateSuccess(clientId, templateId));
            })
            .catch((error) => {
                dispatch(receiveBidTemplateFailed(clientId, templateId, error));
                throw error;
            });
    };
}

// ========== Initialize template =========

function requestTemplate(templateId) {
    return {
        type: REQUEST_TEMPLATE,
        status: {
            message: "Загружаем шаблон",
            status: LoadingStatus.LOADING,
        },
        templateId,
    };
}

function recieveTemplateSuccess(template) {
    return {
        type: RECIEVE_TEMPLATE_SUCCESS,
        status: {
            message: "Шаблон загружен",
            status: LoadingStatus.SUCCESS,
        },
        template,
    };
}

function receiveTemplateFailed(message) {
    return {
        type: RECIEVE_TEMPLATE_FAILED,
        status: {
            message: message,
            status: LoadingStatus.FAILED,
        },
    };
}

function loadTemplate(api, clientId, templateId) {
    return (dispatch) => {
        dispatch(requestTemplate(templateId));
        return api
            .getTemplate(clientId, templateId)
            .then((template) => dispatch(recieveTemplateSuccess(template)))
            .catch((error) => dispatch(receiveTemplateFailed(handleError(error))));
    };
}
