import { uniqueId } from "lodash";

import { getImageSize, handleError } from "./utils";

// ==== API ====

export const REQUEST_UPLOAD_IMAGE = "REQUEST_UPLOAD_IMAGE";
export const RECEIVE_UPLOAD_IMAGE_FAILED = "RECEIVE_UPLOAD_IMAGE_FAILED";
export const RECEIVE_UPLOAD_IMAGE_SUCCESS = "RECEIVE_UPLOAD_IMAGE_SUCCESS";

export const REQUEST_SAVE_CAMPAIGN = "REQUEST_SAVE_CAMPAIGN";
export const RECEIVE_SAVE_CAMPAIGN_FAILED = "RECEIVE_SAVE_CAMPAIGN_FAILED";
export const RECEIVE_SAVE_CAMPAIGN_SUCCESS = "RECEIVE_SAVE_CAMPAIGN_SUCCESS";

export const REQUEST_UPLOAD_CAMPAIGN = "REQUEST_UPLOAD_CAMPAIGN";
export const RECEIVE_UPLOAD_CAMPAIGN_FAILED = "RECEIVE_UPLOAD_CAMPAIGN_FAILED";
export const RECEIVE_UPLOAD_CAMPAIGN_SUCCESS = "RECEIVE_UPLOAD_CAMPAIGN_SUCCESS";

export const SET_SAVE_CAMPAIGN_DRAFT = "SET_SAVE_CAMPAIGN_DRAFT";

// ==== Layout modificators ====

export const ADD_IMAGE = "ADD_IMAGE";

export const ADD_BANNER = "ADD_BANNER";
export const COPY_BANNER = "COPY_BANNER";

export const REMOVE_BANNER = "REMOVE_BANNER";
export const SELECT_IMAGES = "SELECT_IMAGES";

export const ADD_ADGROUP = "ADD_ADGROUP";
export const COPY_ADGROUP = "COPY_ADGROUP";
export const REMOVE_ADGROUP = "REMOVE_ADGROUP";

export const CLEAN_CAMPAIGN = "CLEAN_CAMPAIGN";

// ==== Selectors ====

export const SELECT_COUNTRY = "SELECT_COUNTRY";
export const SELECT_EXPAND_TYPE = "SELECT_EXPAND_TYPE";
export const SELECT_REGIONS = "SELECT_REGIONS";

// ==== Editors ====

export const EDIT_CAMPAIGN_NAME = "EDIT_CAMPAIGN_NAME";
export const EDIT_CAMPAIGN_BID = "EDIT_CAMPAIGN_BID";

export const EDIT_BANNER_TITLES = "EDIT_BANNER_TITLES";
export const EDIT_BANNER_TEXTS = "EDIT_BANNER_TEXTS";
export const EDIT_BANNER_HREFS = "EDIT_BANNER_HREFS";
export const EDIT_BANNER_PARAMS = "EDIT_BANNER_PARAMS";

export const EDIT_ADGROUP_NAME = "EDIT_ADGROUP_NAME";
export const EDIT_ADGROUP_PHRASES = "EDIT_ADGROUP_PHRASES";

export const VALIDATION = "VALIDATION";

// ==== Modals ====

export const SHOW_IMAGE_INFO = "SHOW_IMAGE_INFO";
export const HIDE_IMAGE_INFO = "HIDE_IMAGE_INFO";

export const SHOW_UPLOAD_CAMPAIGN_INFO = "SHOW_UPLOAD_CAMPAIGN_INFO";
export const HIDE_UPLOAD_CAMPAIGN_INFO = "HIDE_UPLOAD_CAMPAIGN_INFO";

export const SHOW_SAVE_CAMPAIGN_INFO = "SHOW_SAVE_CAMPAIGN_INFO";
export const HIDE_SAVE_CAMPAIGN_INFO = "HIDE_SAVE_CAMPAIGN_INFO";

// ==== Layout ====

export const SET_IMAGE_SIZE = "SET_IMAGE_SIZE";

import { validate } from "./validator";
import { APIStatusEnum } from "./utils";

export {
    uploadImages,
    upload,
    showImageInfo,
    hideImageInfo,
    hideSavingInfo,
    hideUploadingInfo,
    showSavingInfo,
    addImage,
    addBanner,
    copyBanner,
    removeBanner,
    addAdGroup,
    copyAdGroup,
    removeAdGroup,
    editCampaignName,
    editCampaignBid,
    editBannerTitles,
    editBannerTexts,
    editBannerHrefs,
    editBannerParams,
    editAdGroupPhrases,
    editAdGroupName,
    selectImages,
    selectCountry,
    selectExpandType,
    selectRegions,
    saveCampaign,
    receiveUploadImageSuccess,
    cleanCampaign,
    receiveSaveCampaignSuccess,
    setImageSize,
    validateForm,
};

// == Images ==

function uploadImage(api, clientId, imageId, image) {
    return (dispatch) => {
        dispatch(requestUploadImage(clientId, imageId));
        return api
            .uploadImage(clientId, image)
            .then((directImage) => dispatch(receiveUploadImageSuccess(clientId, imageId, directImage)))
            .catch((error) => dispatch(receiveUploadImageFailed(clientId, imageId, handleError(error))));
    };
}

function requestUploadImage(clientId, imageId) {
    return {
        type: REQUEST_UPLOAD_IMAGE,
        uploadingStatus: {
            message: "Загружается в директ",
            status: APIStatusEnum.LOADING,
        },
        clientId,
        imageId,
    };
}

function receiveUploadImageSuccess(clientId, imageId, directImage) {
    return {
        type: RECEIVE_UPLOAD_IMAGE_SUCCESS,
        uploadingStatus: {
            message: "Загружено в директ",
            status: APIStatusEnum.SUCCESS,
        },
        direct: directImage,
        clientId,
        imageId,
    };
}

function receiveUploadImageFailed(clientId, imageId, message) {
    return {
        type: RECEIVE_UPLOAD_IMAGE_FAILED,
        uploadingStatus: {
            message: message,
            status: APIStatusEnum.FAILED,
        },
        clientId,
        imageId,
    };
}

// == Saving ==

function buildTemplate(client) {
    let banners = [];
    const images = client.images;
    for (let i = 0; i < client.banners.length; ++i) {
        banners.push({
            uploaded_images: client.banners[i].imageIds.map((imageId) => {
                const imageIndex = images.findIndex((image) => image.id === imageId);
                return images[imageIndex].direct;
            }),
            titles: client.banners[i].titles.filter((value) => value !== ""),
            texts: client.banners[i].texts.filter((value) => value !== ""),
            hrefs: client.banners[i].hrefs.filter((value) => value !== ""),
            href_params: client.banners[i].href_params,
        });
    }
    let adGroups = [];
    for (let i = 0; i < client.adGroups.length; ++i) {
        adGroups.push({
            name: client.adGroups[i].name,
            phrases: client.adGroups[i].phrases.filter((value) => value !== ""),
        });
    }
    return {
        country: client.geo.countryName,
        regions: client.geo.regionIds,
        region_expand_type: client.geo.expandTypeName,
        banner_templates: banners,
        name: client.campaign.name,
        bid: parseInt(client.campaign.bid, 10),
        phrase_groups: adGroups,
    };
}

function saveCampaign(api, client) {
    return (dispatch) => {
        const payload = buildTemplate(client);
        const clientId = client.id;
        dispatch(requestSaveCampaign(clientId));
        let saving_func;
        if (client.savingStatus.status === APIStatusEnum.DRAFT) {
            saving_func = api.createTemplate(clientId, payload);
        } else {
            saving_func = api.updateTemplate(clientId, client.template.id, payload);
        }
        return saving_func
            .then((template) => dispatch(receiveSaveCampaignSuccess(clientId, template)))
            .catch((error) => dispatch(receiveSaveCampaignFailed(clientId, handleError(error))));
    };
}

function setSaveCampaignDraft(clientId) {
    return {
        type: SET_SAVE_CAMPAIGN_DRAFT,
        savingStatus: {
            message: "Кампания создается",
            status: APIStatusEnum.DRAFT,
        },
        clientId,
    };
}

function requestSaveCampaign(clientId) {
    return {
        type: REQUEST_SAVE_CAMPAIGN,
        savingStatus: {
            message: "Кампания отправляется",
            status: APIStatusEnum.LOADING,
        },
        clientId,
    };
}

function receiveSaveCampaignSuccess(clientId, template) {
    return {
        type: RECEIVE_SAVE_CAMPAIGN_SUCCESS,
        savingStatus: {
            message: "Кампания сохранена",
            status: APIStatusEnum.SUCCESS,
        },
        template,
        clientId,
    };
}

function receiveSaveCampaignFailed(clientId, message) {
    return {
        type: RECEIVE_SAVE_CAMPAIGN_FAILED,
        savingStatus: {
            message,
            status: APIStatusEnum.FAILED,
        },
        template: null,
        clientId,
    };
}

// == Uploading ==

function upload(api, client) {
    return (dispatch) => {
        dispatch(sendCampaign(api, client)).then((_) => dispatch(showUploadingInfo(client.id)));
    };
}

function sendCampaign(api, client) {
    return (dispatch) => {
        dispatch(requestUploadCampaign(client.id));
        return api
            .uploadTemplate(client.id, client.template.id)
            .then(() => {
                dispatch(receiveUploadCampaignSuccess(client.id));
                dispatch(setSaveCampaignDraft(client.id));
            })
            .catch((error) => dispatch(receiveUploadCampaignFailed(client.id, handleError(error))));
    };
}

function requestUploadCampaign(clientId) {
    return {
        type: REQUEST_UPLOAD_CAMPAIGN,
        uploadingStatus: {
            message: "Кампания готовится к загрузке в директ",
            status: APIStatusEnum.LOADING,
        },
        clientId,
    };
}

function receiveUploadCampaignSuccess(clientId) {
    return {
        type: RECEIVE_UPLOAD_CAMPAIGN_SUCCESS,
        uploadingStatus: {
            message: "Кампания поставлена в очередь на загрузку. Следите за ней в интерфейсе директа",
            status: APIStatusEnum.SUCCESS,
        },
        clientId,
    };
}

function receiveUploadCampaignFailed(clientId, message) {
    return {
        type: RECEIVE_UPLOAD_CAMPAIGN_FAILED,
        uploadingStatus: {
            message,
            status: APIStatusEnum.FAILED,
        },
        clientId,
    };
}

// ========== Modal ==========

// == Images ==

function showImageInfo(clientId, imageId) {
    return {
        type: SHOW_IMAGE_INFO,
        clientId,
        imageId,
    };
}

function hideImageInfo(clientId) {
    return {
        type: HIDE_IMAGE_INFO,
        clientId,
    };
}

// == Saving ==

function showSavingInfo(clientId) {
    return {
        type: SHOW_SAVE_CAMPAIGN_INFO,
        clientId,
    };
}

function hideSavingInfo(clientId) {
    return {
        type: HIDE_SAVE_CAMPAIGN_INFO,
        clientId,
    };
}

// == Uploading ==

function showUploadingInfo(clientId) {
    return {
        type: SHOW_UPLOAD_CAMPAIGN_INFO,
        clientId,
    };
}

function hideUploadingInfo(clientId) {
    return {
        type: HIDE_UPLOAD_CAMPAIGN_INFO,
        clientId,
    };
}

// ========== Layout modificators ==========

// == Images ==

function addImage(clientId, imageId, image, imageSize) {
    return {
        imageId,
        type: ADD_IMAGE,
        clientId,
        file: image,
        size: imageSize,
    };
}

function setImageSize(clientId, imageId, imageSize) {
    return {
        imageId,
        type: SET_IMAGE_SIZE,
        clientId,
        size: imageSize,
    };
}

function uploadImages(api, clientId, newImages) {
    return (dispatch) => {
        for (let imageIndex = 0; imageIndex < newImages.length; ++imageIndex) {
            const newImage = newImages[imageIndex];
            const newImageId = uniqueId();
            getImageSize(newImage.preview)
                .then((size) => {
                    dispatch(addImage(clientId, newImageId, newImage, size));
                    dispatch(uploadImage(api, clientId, newImageId, newImage));
                })
                .catch((error) => console.error("ERROR", error));
        }
    };
}

// == Banners ==

function addBanner(clientId, bannerId) {
    return {
        type: ADD_BANNER,
        clientId,
        bannerId,
    };
}

function removeBanner(clientId, bannerId) {
    return {
        type: REMOVE_BANNER,
        clientId,
        bannerId,
    };
}

function copyBanner(clientId, fromBannerId, toBannerId) {
    return {
        type: COPY_BANNER,
        clientId,
        fromBannerId,
        toBannerId,
    };
}

// == AdGroups ==

function addAdGroup(clientId, adGroupId) {
    return {
        type: ADD_ADGROUP,
        clientId,
        adGroupId,
    };
}

function removeAdGroup(clientId, adGroupId) {
    return {
        type: REMOVE_ADGROUP,
        clientId,
        adGroupId,
    };
}

function copyAdGroup(clientId, fromAdGroupId, toAdGroupId) {
    return {
        type: COPY_ADGROUP,
        clientId,
        fromAdGroupId,
        toAdGroupId,
    };
}

// ========== Editors ==========

// == Campaign ==

function editCampaignName(clientId, name) {
    return {
        type: EDIT_CAMPAIGN_NAME,
        clientId,
        name,
    };
}

function validateForm(client) {
    return {
        type: VALIDATION,
        clientId: client.id,
        error: validate(client),
    };
}

function editCampaignBid(clientId, bid) {
    return {
        type: EDIT_CAMPAIGN_BID,
        clientId,
        bid,
    };
}

// == Banner ==

function cleanCampaign(clientId) {
    return {
        type: CLEAN_CAMPAIGN,
        clientId,
    };
}

function editBannerTitles(clientId, bannerId, titles) {
    return {
        type: EDIT_BANNER_TITLES,
        clientId,
        bannerId,
        titles,
    };
}

function editBannerTexts(clientId, bannerId, texts) {
    return {
        type: EDIT_BANNER_TEXTS,
        clientId,
        bannerId,
        texts,
    };
}

function editBannerHrefs(clientId, bannerId, hrefs) {
    return {
        type: EDIT_BANNER_HREFS,
        clientId,
        bannerId,
        hrefs,
    };
}

function editBannerParams(clientId, bannerId, paramKey, paramValue) {
    return {
        type: EDIT_BANNER_PARAMS,
        clientId,
        bannerId,
        paramKey,
        paramValue,
    };
}

// == AdGroup ==

function editAdGroupPhrases(clientId, adGroupId, phrases) {
    return {
        type: EDIT_ADGROUP_PHRASES,
        clientId,
        adGroupId,
        phrases,
    };
}

function editAdGroupName(clientId, adGroupId, name) {
    return {
        type: EDIT_ADGROUP_NAME,
        clientId,
        adGroupId,
        name,
    };
}

// ========== Selectors ==========

function selectImages(clientId, bannerId, imageIds) {
    return {
        type: SELECT_IMAGES,
        clientId,
        bannerId,
        imageIds,
    };
}

function selectCountry(clientId, countryName) {
    return (dispatch, getState) => {
        dispatch(_selectCountry(clientId, countryName));
        const countries = getState().countries;
        const countryIndex = countries.findIndex((value) => value.name === countryName);
        const expandTypeIndex = countries[countryIndex].expand_types.findIndex((value) => value.available);
        dispatch(selectExpandType(clientId, countries[countryIndex].expand_types[expandTypeIndex].name));
        dispatch(
            selectRegions(
                clientId,
                countries[countryIndex].regions.map((value) => value.direct_id)
            )
        );
    };
}

function _selectCountry(clientId, countryName) {
    return {
        type: SELECT_COUNTRY,
        clientId,
        countryName,
    };
}

function selectExpandType(clientId, expandTypeName) {
    return {
        type: SELECT_EXPAND_TYPE,
        clientId,
        expandTypeName,
    };
}

function selectRegions(clientId, regionIds) {
    return {
        type: SELECT_REGIONS,
        clientId,
        regionIds,
    };
}
