export const VIEW_UPLOADING_PANEL = "VIEW_UPLOADING_PANEL";
export const VIEW_PREVIEW_PANEL = "VIEW_PREVIEW_PANEL";
export const VIEW_ERROR_PANEL = "VIEW_ERROR_PANEL";
export const VIEW_LOADING_PANEL = "VIEW_LOADING_PANEL";

export const DROP_PREVIEW_PANEL = "DROP_PREVIEW_PANEL";
export const VIEW_CATALOG_PANEL = "VIEW_CATALOG_PANEL";

export const START_INITIALIZATION = "START_INITIALIZATION";
export const END_INITIALIZATION_SUCCESS = "END_INITIALIZATION_SUCCESS";
export const END_INITIALIZATION_FAILED = "END_INITIALIZATION_FAILED";

export const REQUEST_CLIENTS = "REQUEST_CLIENTS";
export const RECEIVE_CLIENTS_SUCCESS = "RECEIVE_CLIENTS_SUCCESS";
export const RECEIVE_CLIENTS_FAILED = "RECEIVE_CLIENTS_FAILED";

export const REQUEST_REGIONS = "REQUEST_REGIONS";
export const RECEIVE_REGIONS_SUCCESS = "RECEIVE_REGIONS_SUCCESS";
export const RECEIVE_REGIONS_FAILED = "RECEIVE_REGIONS_FAILED";

export const SELECT_CLIENT = "SELECT_CLIENT";

import { uniqueId } from "lodash";

import * as Preview from "../actions/Preview";
import * as Upload from "../actions/Upload";
import * as Catalog from "../actions/Catalog";
import {
    getImageSize,
    APIStatusEnum,
    getApiPendingStatus,
    getApiFailedStatus,
    getApiSuccessStatus,
} from "../actions/utils";

export { Preview, Upload, Catalog };

export {
    initializeCatalog,
    initializeUpload,
    startInitialization,
    endInitializationSuccess,
    endInitializationFailed,
    backOnUploadingPanel,
    viewUploadingPanel,
    viewPreviewPanel,
    viewCatalogPanel,
    createTemplate,
    copyTemplate,
    selectClient,
    preview,
    catalog,
    fetchClients,
    fetchRegions,
};

// == Initialization ==

function startInitialization() {
    return {
        type: START_INITIALIZATION,
        initializationStatus: getApiPendingStatus(),
    };
}

function endInitializationSuccess() {
    return {
        type: END_INITIALIZATION_SUCCESS,
        initializationStatus: getApiSuccessStatus(),
    };
}

function endInitializationFailed(error) {
    return {
        type: END_INITIALIZATION_FAILED,
        initializationStatus: getApiFailedStatus(error),
    };
}

// ========== API ==========

function initializeCatalog(api) {
    return (dispatch, getState) => {
        dispatch(viewLoadingPanel());
        dispatch(startInitialization());
        return dispatch(fetchRegions(api))
            .then(() => dispatch(fetchClients(api)))
            .then(() => dispatch(Catalog.fetchTemplates(api, getState().activeClientId)))
            .then(() => {
                dispatch(endInitializationSuccess());
                dispatch(viewCatalogPanel());
            })
            .catch((error) => {
                dispatch(endInitializationFailed(error));
                dispatch(viewErrorPanel());
            });
    };
}

// == Clients ==

function fetchClients(api) {
    return (dispatch) => {
        dispatch(requestClients());
        return api
            .getClients()
            .then((clients) => {
                dispatch(receiveClientsSuccess(clients));
                dispatch(selectClient(clients[0].direct_id));
                clients.forEach((client) => {
                    dispatch(initializeByDefault(client.direct_id));
                });
            })
            .catch((error) => {
                dispatch(receiveClientsFailed(error));
                throw error;
            });
    };
}

function requestClients() {
    return {
        type: REQUEST_CLIENTS,
        clientsStatus: getApiPendingStatus(),
    };
}

function receiveClientsSuccess(clients) {
    return {
        type: RECEIVE_CLIENTS_SUCCESS,
        clients: clients,
        clientsStatus: getApiSuccessStatus(),
    };
}

function receiveClientsFailed(error) {
    return {
        type: RECEIVE_CLIENTS_FAILED,
        clientsStatus: getApiFailedStatus(error),
    };
}

function selectClient(clientId) {
    return {
        type: SELECT_CLIENT,
        clientId,
    };
}

// == Regions ==

function fetchRegions(api) {
    return (dispatch) => {
        dispatch(requestRegions());
        return api
            .getRegions()
            .then((regions) => {
                const action = receiveRegionsSuccess(regions);
                dispatch(action);
            })
            .catch((error) => {
                dispatch(receiveRegionsFailed(error));
                throw error;
            });
    };
}

function requestRegions() {
    return {
        type: REQUEST_REGIONS,
        countriesStatus: getApiPendingStatus(),
    };
}

function receiveRegionsSuccess(regions) {
    return {
        type: RECEIVE_REGIONS_SUCCESS,
        countries: regions,
        countriesStatus: getApiSuccessStatus(),
    };
}

function receiveRegionsFailed(error) {
    return {
        type: RECEIVE_REGIONS_FAILED,
        countriesStatus: getApiFailedStatus(error),
    };
}

// == Views ==

function viewUploadingPanel() {
    return {
        type: VIEW_UPLOADING_PANEL,
    };
}

function viewPreviewPanel() {
    return {
        type: VIEW_PREVIEW_PANEL,
    };
}

function viewCatalogPanel() {
    return {
        type: VIEW_CATALOG_PANEL,
    };
}

function viewErrorPanel() {
    return {
        type: VIEW_ERROR_PANEL,
    };
}

function viewLoadingPanel() {
    return {
        type: VIEW_LOADING_PANEL,
    };
}

function catalog(api, clientId) {
    return (dispatch) => {
        dispatch(Catalog.fetchTemplates(api, clientId)).then(() => dispatch(viewCatalogPanel()));
    };
}

function backOnUploadingPanel() {
    return (dispatch) => {
        dispatch(dropPreviewPanel());
        dispatch(viewUploadingPanel());
    };
}

function dropPreviewPanel() {
    return {
        type: DROP_PREVIEW_PANEL,
    };
}

function preview(api, client) {
    return (dispatch, getState) => {
        dispatch(Upload.validateForm(client));
        const clients = getState().clients;
        const clientIndex = clients.findIndex((value) => value.id === client.id);
        if (clients[clientIndex].validation === null) {
            dispatch(Upload.saveCampaign(api, client)).then(() => {
                const clients = getState().clients;
                const clientIndex = clients.findIndex((value) => value.id === client.id);
                if (clients[clientIndex].savingStatus.status === APIStatusEnum.SUCCESS) {
                    return dispatch(viewPreviewPanel());
                } else {
                    return dispatch(Upload.showSavingInfo(client.id));
                }
            });
        }
    };
}

function copyTemplate(api, clientId, templateId) {
    return (dispatch, getState) => {
        dispatch(Catalog.loadTemplate(api, clientId, templateId)).then(() => {
            dispatch(Upload.cleanCampaign(clientId));
            dispatch(initializeUpload(clientId, getState().catalogPanel.selectedTemplate));
            dispatch(viewUploadingPanel());
        });
    };
}

function createTemplate(clientId) {
    return (dispatch) => {
        dispatch(Upload.cleanCampaign(clientId));
        dispatch(initializeByDefault(clientId));
        dispatch(viewUploadingPanel());
    };
}

function initializeByDefault(clientId) {
    return (dispatch, getState) => {
        const defaultTemplate = {
            name: "",
            phrase_groups: [
                {
                    phrases: [],
                    name: "",
                },
            ],
            country: getState().countries[0].name,
            region_expand_type: null,
            bid: "",
            regions: null,
            banner_templates: [
                {
                    hrefs: [],
                    texts: [],
                    titles: [],
                    uploaded_images: [],
                    href_params: {
                        utm_campaign: "",
                        utm_medium: "",
                        utm_source: "",
                        utm_term: "",
                    },
                },
            ],
        };
        return dispatch(initializeUpload(clientId, defaultTemplate));
    };
}

function initializeUpload(clientId, template) {
    return (dispatch) => {
        dispatch(Upload.editCampaignName(clientId, template.name));
        dispatch(Upload.editCampaignBid(clientId, String(template.bid)));
        dispatch(Upload.selectCountry(clientId, template.country));

        if (template.region_expand_type !== null) {
            dispatch(Upload.selectExpandType(clientId, template.region_expand_type));
        }
        if (template.regions !== null) {
            dispatch(Upload.selectRegions(clientId, template.regions));
        }

        const hashes = {};
        template.banner_templates.forEach((banner_template) => {
            const bannerId = uniqueId();
            dispatch(Upload.addBanner(clientId, bannerId));
            dispatch(Upload.editBannerTitles(clientId, bannerId, banner_template.titles));
            dispatch(Upload.editBannerTexts(clientId, bannerId, banner_template.texts));
            dispatch(Upload.editBannerHrefs(clientId, bannerId, banner_template.hrefs));
            Object.keys(banner_template.href_params).forEach((key) => {
                dispatch(Upload.editBannerParams(clientId, bannerId, key, banner_template.href_params[key] || ""));
            });
            const images = [];
            banner_template.uploaded_images.forEach((directImage) => {
                if (!(directImage.hash in hashes)) {
                    const imageId = uniqueId();
                    hashes[directImage.hash] = imageId;
                    dispatch(Upload.addImage(clientId, imageId, null));
                    getImageSize(directImage.url).then((size) =>
                        dispatch(Upload.setImageSize(clientId, imageId, size))
                    );
                    dispatch(Upload.receiveUploadImageSuccess(clientId, imageId, directImage));
                    images.push(imageId);
                } else {
                    images.push(hashes[directImage.hash]);
                }
            });
            dispatch(Upload.selectImages(clientId, bannerId, images));
        });
        template.phrase_groups.forEach((adGroup) => {
            const adGroupId = uniqueId();
            dispatch(Upload.addAdGroup(clientId, adGroupId));
            dispatch(Upload.editAdGroupName(clientId, adGroupId, adGroup.name));
            dispatch(Upload.editAdGroupPhrases(clientId, adGroupId, adGroup.phrases));
        });
    };
}
