const WindowsEnum = Object.freeze({
    UPLOAD: 0,
    PREVIEW: 1,
    CATALOG: 2,
    ERROR: 3,
    LOADING: 4,
});
const APIStatusEnum = Object.freeze({
    DRAFT: 0,
    LOADING: 1,
    SUCCESS: 2,
    FAILED: 3,
});
const APIStatusMessage = Object.freeze({
    DRAFT: "Данные еще не начали загружаться",
    LOADING: "Данные загружаются",
    SUCCESS: "Данные загружены",
    FAILED: "Данные не удалось загрузить",
});

function getImageSize(image) {
    return new Promise(function (resolve, reject) {
        const img = document.createElement("img");
        img.onload = function onload() {
            resolve({ width: img.width, height: img.height });
        };
        img.onerror = function onerror(err) {
            reject(err);
        };
        img.src = image;
    });
}

function handleError(error) {
    let message = "";
    if (error.status === 400) {
        message = error.obj.message + ". Errors: ";
        for (let e in error.obj.errors) {
            if (error.obj.errors.hasOwnProperty(e)) {
                message += "( " + e + ": " + error.obj.errors[e] + " )";
            }
        }
    } else if (error.status === 419 && error.obj.error_code === 11004) {
        console.error(error);
        message = "Вы пытаетесь повторно загрузить ту же кампанию";
    } else {
        console.error(error);
        message = "Внутренняя ошибка сервера(" + String(error.status) + ")";
    }
    return message;
}

function getApiFailedStatus(error) {
    let message = "";
    let errors = [];
    if (error.status === 400) {
        message = error.obj.message;
        if (error.obj.errors) {
            errors = error.obj.errors;
        }
    } else {
        console.error(error);
        message = "Внутренняя ошибка сервера(" + String(error.status) + ")";
    }
    return {
        message: message || APIStatusMessage.FAILED,
        errors,
        code: error.status,
        status: APIStatusEnum.FAILED,
    };
}

function getApiSuccessStatus(message) {
    return {
        message: message || APIStatusMessage.SUCCESS,
        errors: [],
        code: 200,
        status: APIStatusEnum.SUCCESS,
    };
}

function getApiPendingStatus() {
    return {
        message: APIStatusMessage.LOADING,
        errors: [],
        code: 0,
        status: APIStatusEnum.LOADING,
    };
}

function getApiDraftStatus() {
    return {
        message: APIStatusMessage.DRAFT,
        errors: [],
        code: 0,
        status: APIStatusEnum.DRAFT,
    };
}

function splitTextField(text) {
    return text !== "" ? text.split("\n") : [];
}

export {
    getImageSize,
    handleError,
    splitTextField,
    APIStatusEnum,
    APIStatusMessage,
    WindowsEnum,
    getApiFailedStatus,
    getApiPendingStatus,
    getApiSuccessStatus,
    getApiDraftStatus,
};
