import FieldVal, { BasicVal } from "fieldval";

// http://www.fieldval.com/docs/fieldval/Errors
const ValidationErrorTypes = Object.freeze({
    MULTIPLE_ERRORS: 4,
    ONE_OR_MORE_ERRORS: 5,
});

const bidLessThenZeroError = () => ({
    error: 1001,
    error_message: "Задайте ставку больше нуля",
});

const bidNotIntegerError = () => ({
    error: 1002,
    error_message: "Задайте ставку целым числом",
});

const bidIsEmptyError = () => ({
    error: 1003,
    error_message: "Задайте ставку",
});

const campaignNameIsEmptyError = () => ({
    error: 1004,
    error_message: "Задайте имя кампании",
});

const campaignNameIsTooLargeError = (limit) => ({
    error: 1005,
    error_message: "Задайте имя кампании короче " + limit + " символов",
});

const adGroupsIsEmptyError = () => ({
    error: 1006,
    error_message: "Добавьте группу фраз",
});

const adGroupNameIsEmptyError = (adGroupIndex) => ({
    error: 1007,
    error_message: "Задайте имя группе #" + (adGroupIndex + 1),
});

const adGroupNameIsTooLargeError = (adGroupIndex, limit) => ({
    error: 1008,
    error_message: "Имя группы #'" + adGroupIndex + "' длиннее  " + limit + " символов",
});

const phrasesIsEmptyError = (adGroupIndex) => ({
    error: 1009,
    error_message: "Добавьте фразы в группу #" + (adGroupIndex + 1),
});

const phraseIsEmptyError = (adGroupIndex, phraseIndex) => ({
    error: 1010,
    error_message: "В группе #" + (adGroupIndex + 1) + " фраза # " + (phraseIndex + 1) + " пустая",
});

const phraseIsTooLargeError = (adGroupIndex, phraseIndex, limit) => ({
    error: 1011,
    error_message:
        "В группе #" + (adGroupIndex + 1) + " фраза # " + (phraseIndex + 1) + " длиннее " + limit + " символов",
});

const bannersIsEmptyError = () => ({
    error: 1012,
    error_message: "Добавьте баннер",
});

const titlesIsEmptyError = (bannerIndex) => ({
    error: 1013,
    error_message: "Добавьте заголовки в баннер #" + (bannerIndex + 1),
});

const textsIsEmptyError = (bannerIndex) => ({
    error: 1014,
    error_message: "Добавьте тексты в баннер #" + (bannerIndex + 1),
});

const hrefsIsEmptyError = (bannerIndex) => ({
    error: 1015,
    error_message: "Добавьте урлы в баннер #" + (bannerIndex + 1),
});

const titleIsEmptyError = (bannerIndex, titleIndex) => ({
    error: 1016,
    error_message: "В баннере #" + (bannerIndex + 1) + " заголовок #" + (titleIndex + 1) + " пустой",
});

const textIsEmptyError = (bannerIndex, textIndex) => ({
    error: 1017,
    error_message: "В баннере #" + (bannerIndex + 1) + " текст #" + (textIndex + 1) + " пустой",
});

const hrefIsEmptyError = (bannerIndex, hrefIndex) => ({
    error: 1018,
    error_message: "В баннере #" + (bannerIndex + 1) + " урл #" + (hrefIndex + 1) + " пустой",
});

const titleIsTooLargeError = (bannerIndex, titleIndex, limit) => ({
    error: 1019,
    error_message:
        "В баннере #" + (bannerIndex + 1) + " заголовок #" + (titleIndex + 1) + " длиннее " + limit + " символов",
});

const textIsTooLargeError = (bannerIndex, textIndex, limit) => ({
    error: 1020,
    error_message: "В баннере #" + (bannerIndex + 1) + " текст #" + (textIndex + 1) + " длиннее " + limit + " символов",
});

const hrefIsTooLargeError = (bannerIndex, hrefIndex, limit) => ({
    error: 1021,
    error_message: "В баннере #" + (bannerIndex + 1) + " урл #" + (hrefIndex + 1) + " длиннее " + limit + " символов",
});

const hrefIsNotValidError = (bannerIndex, hrefIndex) => ({
    error: 1022,
    error_message: "В баннере #" + (bannerIndex + 1) + " урл #" + (hrefIndex + 1) + " невалидный",
});

const hrefIsNotStartedByHttpError = (bannerIndex, hrefIndex) => ({
    error: 1023,
    error_message: "В баннере #" + (bannerIndex + 1) + " урл #" + (hrefIndex + 1) + " начинается не с http/https",
});

const regionsIsEmptyError = () => ({
    error: 1024,
    error_message: "Выберите хотя бы один регион",
});

function validate(client) {
    const data = {
        name: client.campaign.name,
        bid: client.campaign.bid,
        adGroups: client.adGroups,
        banners: client.banners,
        regions: client.geo.regionIds,
    };
    const validator = new FieldVal(data);

    validator.get(
        "name",
        BasicVal.not_empty(true, { error: campaignNameIsEmptyError() }),
        BasicVal.max_length(150, { error: campaignNameIsTooLargeError(150) })
    );

    validator.get(
        "bid",
        BasicVal.not_empty(true, { error: bidIsEmptyError() }),
        BasicVal.integer(true, { parse: true, error: bidNotIntegerError() }),
        BasicVal.minimum(1, { parse: true, error: bidLessThenZeroError() })
    );

    validator.get(
        "adGroups",
        BasicVal.min_length(1, { error: adGroupsIsEmptyError() }),
        BasicVal.each(function (adGroup, adGroupIndex, emit) {
            const adGroupData = {
                name: adGroup.name,
                phrases: adGroup.phrases,
            };
            const new_validator = new FieldVal(adGroupData);

            new_validator.get(
                "name",
                BasicVal.not_empty(true, {
                    error: adGroupNameIsEmptyError(adGroupIndex),
                }),
                BasicVal.max_length(150, {
                    error: adGroupNameIsTooLargeError(adGroupIndex, 150),
                })
            );
            new_validator.get(
                "phrases",
                BasicVal.min_length(1, {
                    error: phrasesIsEmptyError(adGroupIndex),
                }),
                BasicVal.each(function (phrase, phraseIndex) {
                    return (
                        BasicVal.not_empty(true, {
                            error: phraseIsEmptyError(adGroupIndex, phraseIndex),
                        }).check(phrase) ||
                        BasicVal.max_length(4096, {
                            error: phraseIsTooLargeError(adGroupIndex, phraseIndex, 4096),
                        }).check(phrase)
                    );
                })
            );
            return new_validator.end();
        })
    );

    validator.get(
        "banners",
        BasicVal.min_length(1, { error: bannersIsEmptyError() }),
        BasicVal.each(function (banner, bannerIndex) {
            const bannerData = {
                titles: banner.titles,
                texts: banner.texts,
                hrefs: banner.hrefs,
            };
            const new_validator = new FieldVal(bannerData);

            new_validator.get(
                "titles",
                BasicVal.min_length(1, {
                    error: titlesIsEmptyError(bannerIndex),
                }),
                BasicVal.each(function (title, titleIndex) {
                    return (
                        BasicVal.not_empty(true, {
                            error: titleIsEmptyError(bannerIndex, titleIndex),
                        }).check(title) ||
                        BasicVal.max_length(33, {
                            error: titleIsTooLargeError(bannerIndex, titleIndex, 33),
                        }).check(title)
                    );
                })
            );
            new_validator.get(
                "texts",
                BasicVal.min_length(1, {
                    error: textsIsEmptyError(bannerIndex),
                }),
                BasicVal.each(function (text, textIndex) {
                    return (
                        BasicVal.not_empty(true, {
                            error: textIsEmptyError(bannerIndex, textIndex),
                        }).check(text) ||
                        BasicVal.max_length(75, {
                            error: textIsTooLargeError(bannerIndex, textIndex, 75),
                        }).check(text)
                    );
                })
            );
            new_validator.get(
                "hrefs",
                BasicVal.min_length(1, {
                    error: hrefsIsEmptyError(bannerIndex),
                }),
                BasicVal.each(function (href, hrefIndex) {
                    return (
                        BasicVal.not_empty(true, {
                            error: hrefIsEmptyError(bannerIndex, hrefIndex),
                        }).check(href) ||
                        BasicVal.prefix("http", {
                            error: hrefIsNotStartedByHttpError(bannerIndex, hrefIndex),
                        }).check(href) ||
                        BasicVal.url(true, {
                            error: hrefIsNotValidError(bannerIndex, hrefIndex),
                        }).check(href) ||
                        BasicVal.max_length(1024, {
                            error: hrefIsTooLargeError(bannerIndex, hrefIndex, 1024),
                        }).check(href)
                    );
                })
            );
            return new_validator.end();
        })
    );

    validator.get("regions", BasicVal.array(true), BasicVal.min_length(1, { error: regionsIsEmptyError() }));

    return validator.end();
}

export { validate, ValidationErrorTypes };
