import React from "react";
import PropTypes from "prop-types";
import { connect, Provider } from "react-redux";
import { createStore, applyMiddleware, compose } from "redux";
import thunk from "redux-thunk";

import DirectLoaderReducer from "./reducers/index";

const composeEnhancers = window.__REDUX_DEVTOOLS_EXTENSION_COMPOSE__ || compose;
export let store = createStore(DirectLoaderReducer, composeEnhancers(applyMiddleware(thunk)));

import { Spin } from "@yandex-lego/components/Spin/desktop/bundle";

import PreviewPanel from "./PreviewPanel";
import UploadingPanel from "./UploadingPanel";
import CatalogPanel from "./CatalogPanel";

import "./index.scss";

import { initializeUpload, initializeCatalog } from "./actions";
import { WindowsEnum } from "./actions/utils";

const IDM_LINK =
    "https://idm.yandex-team.ru/system/crypta/roles#rf=1,rf-role=uv5y6qYP#crypta/distribution;;;,f-status=all,f-role=crypta,sort-by=-updated,rf-expanded=uv5y6qYP";
const TELEGRAM_LINK = "https://t.me/joinchat/AAAAAELNcH5zenrJS5YQGw";
const WIKI_LINK = "https://wiki.yandex-team.ru/bannernajakrutilka/distribucija/DistributionPhrasesFactory/Geofabrika";

class DirectLoaderPanelBase extends React.Component {
    componentDidMount() {
        const { initUpload, initCatalog, selectedTemplate, selectedClient } = this.props;
        initCatalog();
        if (selectedTemplate !== null) {
            initUpload(selectedClient.id, selectedTemplate);
        }
    }

    renderUpload() {
        const { api, selectedClient, clients } = this.props;
        return <UploadingPanel api={api} client={selectedClient} clients={clients} />;
    }

    renderPreview() {
        const { api, selectedClient } = this.props;

        return <PreviewPanel api={api} clientId={selectedClient.id} />;
    }

    renderCatalog() {
        const { api, selectedClient, templates, clients } = this.props;
        return <CatalogPanel api={api} clientId={selectedClient.id} templates={templates} clients={clients} />;
    }

    renderError(message) {
        return (
            <div className="ErrorContainer">
                <div className="YandexSansText-Medium ErrorMessage" dangerouslySetInnerHTML={{ __html: message }} />
                <div className="ErrorPicture" />
            </div>
        );
    }

    renderCommonError() {
        let message = `Геодирект недоступен. Попробуйте позже.
                       <br/>
                       Если не заработал - <a href="${TELEGRAM_LINK}">напишите нам в телеграм</a>`;
        return this.renderError(message);
    }

    renderAuthorizationError() {
        let message = `У вас нет доступа к геодиректу
                      <br/>
                      Запросите <a href="${IDM_LINK}">в IDM</a> роль "Дистрибуция"
                      <br/>
                      Про возможности Геодиректа можете прочитать на <a href="${WIKI_LINK}">вики</a>`;
        return this.renderError(message);
    }

    render() {
        const { window, errorCode } = this.props;
        if (window === WindowsEnum.UPLOAD) {
            return <div className="LayoutFinish DirectLoaderLayout">{this.renderUpload()}</div>;
        } else if (window === WindowsEnum.PREVIEW) {
            return <div className="LayoutFinish DirectLoaderLayout">{this.renderPreview()}</div>;
        } else if (window === WindowsEnum.CATALOG) {
            return <div className="LayoutFinish DirectLoaderLayout">{this.renderCatalog()}</div>;
        } else if (window === WindowsEnum.ERROR) {
            let error;
            if (errorCode === 403) {
                error = this.renderAuthorizationError();
            } else {
                error = this.renderCommonError();
            }

            return <div className="LayoutFinish">{error}</div>;
        } else if (window === WindowsEnum.LOADING) {
            return (
                <div className="loading">
                    <Spin progress={true} view="default" size="l" />
                </div>
            );
        } else {
            return <div className="LayoutFinish">{this.renderCommonError()}</div>;
        }
    }
}

function getClient(state, clientId) {
    const clientIndex = state.clients.findIndex((value) => value.id === clientId);
    return state.clients[clientIndex];
}

function mapStateToProps(state, ownProps) {
    return {
        window: state.window,
        selectedClient: getClient(state, state.activeClientId),
        selectedTemplate: state.catalogPanel.selectedTemplate,
        templates: state.catalogPanel.templates,
        clients: state.clients,
        errorCode: state.initializationStatus.code,
    };
}

function mapDispatchToProps(dispatch, ownProps) {
    return {
        initCatalog: (clientId) => dispatch(initializeCatalog(ownProps.api)),
        initUpload: (clientId, template) => dispatch(initializeUpload(clientId, template)),
    };
}

DirectLoaderPanelBase.propTypes = {
    api: PropTypes.object.isRequired,
};

const DirectLoaderPanel = connect(mapStateToProps, mapDispatchToProps)(DirectLoaderPanelBase);

class DirectLoader extends React.Component {
    render() {
        return (
            <Provider store={store}>
                <DirectLoaderPanel api={this.props.phf_api} />
            </Provider>
        );
    }
}

const outerMapStateToProps = (state) => ({
    phf_api: state.base.phf_api,
});

export default connect(outerMapStateToProps)(DirectLoader);
export { DirectLoaderPanelBase, DirectLoaderPanel };
