// ==== Reducers structure ====
//
// initial functions (like initialTemplate(), initialCatalogPanel() are provided for initialization state even if they are created without root reducer
// initialXXX() property <undefined> - this property has its own reducer inside XXXReducer()
// initialXXX() property <null> - this property must be updated by API call
// initialXXX() property <some empty value, i.e. [], ""> - this property does not be necessarily redefined
//

import update from "immutability-helper";
import { uniqueId } from "lodash";

import * as Actions from "../actions";
import { APIStatusEnum, APIStatusMessage, WindowsEnum, getApiDraftStatus } from "../actions/utils";

// ===== Templates Reducer

const initialTemplate = () => ({
    id: uniqueId(),
    name: "",
    creationTime: null,
    wasUploaded: false,
    taskStatus: {
        statusName: "Создание шаблона",
        statusId: -1,
    },
    task: null,
    moderationStatus: getApiDraftStatus(),
    biddingStatus: getApiDraftStatus(),
});

function templatesReducer(state = [], action) {
    let templateIndex = null;
    switch (action.type) {
        case Actions.Catalog.REQUEST_BIDDING:
        case Actions.Catalog.RECEIVE_BIDDING_SUCCESS:
        case Actions.Catalog.RECEIVE_BIDDING_FAILED:
            templateIndex = state.findIndex((value) => value.id === action.templateId);
            return update(state, {
                [templateIndex]: {
                    biddingStatus: { $set: action.biddingStatus },
                },
            });
        case Actions.Catalog.REQUEST_MODERATION:
        case Actions.Catalog.RECEIVE_MODERATION_SUCCESS:
        case Actions.Catalog.RECEIVE_MODERATION_FAILED:
            templateIndex = state.findIndex((value) => value.id === action.templateId);
            return update(state, {
                [templateIndex]: {
                    moderationStatus: { $set: action.moderationStatus },
                },
            });
        default:
            return state;
    }
}

// ===== CatalogPanel Reducer ======

const initialCatalogPanel = {
    selectedTemplate: null,
    selectedTemplateStatus: getApiDraftStatus(),
    templates: undefined,
    templatesStatus: getApiDraftStatus(),
};

function catalogPanelReducer(state = initialCatalogPanel, action) {
    const intermediateState = {
        selectedTemplate: state.selectedTemplate,
        selectedTemplateStatus: state.selectedTemplateStatus,
        templates: templatesReducer(state.templates, action),
        templatesStatus: state.templatesStatus,
    };

    switch (action.type) {
        case Actions.Catalog.REQUEST_TEMPLATES:
        case Actions.Catalog.RECEIVE_TEMPLATES_FAILED:
            return update(intermediateState, {
                templatesStatus: { $set: action.templatesStatus },
            });
        case Actions.Catalog.RECEIVE_TEMPLATES_SUCCESS:
            const updatedTemplates = action.templates.map((template) =>
                update(initialTemplate(), {
                    id: { $set: template.id },
                    name: { $set: template.name },
                    creationTime: { $set: template.creation_time },
                    wasUploaded: { $set: template.was_uploaded },
                    task: { $set: template.last_task_name },
                    taskStatus: {
                        statusId: { $set: template.last_task_state.status_id },
                        statusName: {
                            $set: template.last_task_state.status_name,
                        },
                    },
                })
            );
            return update(intermediateState, {
                templates: { $set: templatesReducer(updatedTemplates, action) },
                templatesStatus: { $set: action.templatesStatus },
            });
        case Actions.Catalog.REQUEST_TEMPLATE:
            return update(intermediateState, {
                selectedTemplateStatus: { $set: action.status },
                selectedTemplate: { $set: { id: action.templateId } },
            });
        case Actions.Catalog.RECIEVE_TEMPLATE_SUCCESS:
            return update(intermediateState, {
                selectedTemplateStatus: { $set: action.status },
                selectedTemplate: { $merge: action.template },
            });
        case Actions.Catalog.RECIEVE_TEMPLATE_FAILED:
            return update(intermediateState, {
                selectedTemplateStatus: { $set: action.status },
            });
        default:
            return intermediateState;
    }
}

// ===== PreviewPanel Reducer ======

const initialPreviewPanel = {
    isLoaded: false,
    campaigns: {
        values: [],
        active: null,
    },
    adGroups: {
        values: [],
        active: null,
    },
    banners: [],
    phrases: [],
};

function previewPanelReducer(state = initialPreviewPanel, action) {
    switch (action.type) {
        case Actions.DROP_PREVIEW_PANEL:
            return update(state, { isLoaded: { $set: false } });
        case Actions.Preview.LOADING_PREVIEW_PANEL_DONE:
            return update(state, { isLoaded: { $set: true } });
        case Actions.Preview.RECEIVE_PREVIEW_CAMPAIGNS:
            return update(state, {
                campaigns: {
                    values: { $set: action.campaigns },
                    active: { $set: action.activeId },
                },
            });
        case Actions.Preview.RECEIVE_PREVIEW_ADGROUPS:
            return update(state, {
                adGroups: {
                    values: { $set: action.adGroups },
                    active: { $set: action.activeId },
                },
            });
        case Actions.Preview.RECEIVE_PREVIEW_PHRASES:
            const phrases = action.phrases.map((phrase) => {
                return { id: uniqueId(), name: phrase };
            });
            return update(state, { phrases: { $set: phrases } });
        case Actions.Preview.RECEIVE_PREVIEW_BANNERS:
            const banners = action.banners.map((banner) => update(banner, { $merge: { id: uniqueId() } }));
            return update(state, { banners: { $set: banners } });
        case Actions.Preview.SELECT_PREVIEW_ADGROUP:
            return update(state, {
                adGroups: { active: { $set: action.adGroupId } },
            });
        case Actions.Preview.SELECT_PREVIEW_CAMPAIGN:
            return update(state, {
                campaigns: { active: { $set: action.campaignId } },
            });
        default:
            return state;
    }
}

// ===== Client Reducer ======

const initialClient = () => ({
    id: null,
    name: null,
    images: undefined,
    infoViewImageId: null,
    adGroups: undefined,
    banners: undefined,
    geo: undefined,
    campaign: undefined,
    savingStatus: {
        status: APIStatusEnum.DRAFT,
        message: APIStatusMessage.DRAFT,
        errors: [],
        visible: false,
        code: 0,
    },
    uploadingStatus: {
        status: APIStatusEnum.DRAFT,
        message: APIStatusMessage.DRAFT,
        errors: [],
        visible: false,
        code: 0,
    },
    template: null,
    validation: null,
});

function clientReducer(state = initialClient(), action) {
    if (action.clientId !== undefined && state.id !== action.clientId) {
        return state;
    }

    const intermediateState = {
        id: state.id,
        name: state.name,
        images: imagesReducer(state.images, action),
        infoViewImageId: state.infoViewImageId || null,
        adGroups: adGroupsReducer(state.adGroups, action),
        banners: bannersReducer(state.banners, action),
        geo: geoReducer(state.geo, action),
        campaign: campaignReducer(state.campaign, action),
        savingStatus: state.savingStatus,
        uploadingStatus: state.uploadingStatus,
        template: state.template,
        validation: state.validation,
    };

    switch (action.type) {
        case Actions.Upload.VALIDATION:
            return update(intermediateState, {
                validation: { $set: action.error },
            });
        case Actions.Upload.CLEAN_CAMPAIGN:
            return update(initialClient(), {
                id: { $set: state.id },
                name: { $set: state.name },
            });
        case Actions.Upload.SET_SAVE_CAMPAIGN_DRAFT:
        case Actions.Upload.REQUEST_SAVE_CAMPAIGN:
            return update(intermediateState, {
                savingStatus: { $merge: action.savingStatus },
            });
        case Actions.Upload.RECEIVE_SAVE_CAMPAIGN_FAILED:
        case Actions.Upload.RECEIVE_SAVE_CAMPAIGN_SUCCESS:
            return update(intermediateState, {
                savingStatus: { $merge: action.savingStatus },
                template: { $set: action.template },
            });
        case Actions.Upload.REQUEST_UPLOAD_CAMPAIGN:
        case Actions.Upload.RECEIVE_UPLOAD_CAMPAIGN_FAILED:
        case Actions.Upload.RECEIVE_UPLOAD_CAMPAIGN_SUCCESS:
            return update(intermediateState, {
                uploadingStatus: { $merge: action.uploadingStatus },
            });
        case Actions.Upload.SHOW_SAVE_CAMPAIGN_INFO:
            return update(intermediateState, {
                savingStatus: { visible: { $set: true } },
            });
        case Actions.Upload.HIDE_SAVE_CAMPAIGN_INFO:
            return update(intermediateState, {
                savingStatus: { visible: { $set: false } },
            });
        case Actions.Upload.SHOW_UPLOAD_CAMPAIGN_INFO:
            return update(intermediateState, {
                uploadingStatus: { visible: { $set: true } },
            });
        case Actions.Upload.HIDE_UPLOAD_CAMPAIGN_INFO:
            return update(intermediateState, {
                uploadingStatus: { visible: { $set: false } },
            });
        case Actions.Upload.SHOW_IMAGE_INFO:
            return update(intermediateState, {
                infoViewImageId: { $set: action.imageId },
            });
        case Actions.Upload.HIDE_IMAGE_INFO:
            return update(intermediateState, {
                infoViewImageId: { $set: null },
            });
        default:
            return intermediateState;
    }
}

// ===== Images Reducer ======

const initialImage = () => ({
    id: uniqueId(),
    file: null,
    direct: {
        url: null,
        hash: null,
    },
    uploadingStatus: getApiDraftStatus(),
    size: {
        width: null,
        height: null,
    },
});

function imagesReducer(state = [], action) {
    let imageIndex;
    switch (action.type) {
        case Actions.Upload.ADD_IMAGE:
            const newImage = update(initialImage(), {
                id: { $set: action.imageId },
                file: { $set: action.file },
                size: { $set: action.size },
            });
            return update(state, { $push: [newImage] });
        case Actions.Upload.SET_IMAGE_SIZE:
            imageIndex = state.findIndex((value) => value.id === action.imageId);
            return update(state, {
                [imageIndex]: {
                    size: { $set: action.size },
                },
            });
        case Actions.Upload.REQUEST_UPLOAD_IMAGE:
        case Actions.Upload.RECEIVE_UPLOAD_IMAGE_FAILED:
            imageIndex = state.findIndex((value) => value.id === action.imageId);
            return update(state, {
                [imageIndex]: {
                    uploadingStatus: { $set: action.uploadingStatus },
                },
            });
        case Actions.Upload.RECEIVE_UPLOAD_IMAGE_SUCCESS:
            imageIndex = state.findIndex((value) => value.id === action.imageId);
            return update(state, {
                [imageIndex]: {
                    uploadingStatus: { $set: action.uploadingStatus },
                    direct: {
                        url: { $set: action.direct.url },
                        hash: { $set: action.direct.hash },
                    },
                },
            });
        default:
            return state;
    }
}

// ===== AdGroup Reducer ======

const initialAdGroup = () => ({
    id: uniqueId(),
    name: "",
    phrases: [],
});

function adGroupsReducer(state = [], action) {
    let adGroupIndex = null;
    switch (action.type) {
        case Actions.Upload.EDIT_ADGROUP_NAME:
        case Actions.Upload.EDIT_ADGROUP_PHRASES:
            adGroupIndex = state.findIndex((value) => value.id === action.adGroupId);
            return update(state, {
                [adGroupIndex]: {
                    $set: adGroupReducer(state[adGroupIndex], action),
                },
            });
        case Actions.Upload.ADD_ADGROUP:
            const newAdGroup = update(initialAdGroup(), {
                id: { $set: action.adGroupId },
            });
            return update(state, { $push: [newAdGroup] });
        case Actions.Upload.COPY_ADGROUP:
            adGroupIndex = state.findIndex((value) => value.id === action.fromAdGroupId);
            const copiedAdGroup = update(state[adGroupIndex], {
                id: { $set: action.toAdGroupId },
            });
            return update(state, { $push: [copiedAdGroup] });
        case Actions.Upload.REMOVE_ADGROUP:
            adGroupIndex = state.findIndex((value) => value.id === action.adGroupId);
            return update(state, { $splice: [[adGroupIndex, 1]] });
        default:
            return state;
    }
}

function adGroupReducer(state = initialAdGroup(), action) {
    switch (action.type) {
        case Actions.Upload.EDIT_ADGROUP_NAME:
            return update(state, { name: { $set: action.name } });
        case Actions.Upload.EDIT_ADGROUP_PHRASES:
            return update(state, { phrases: { $set: action.phrases } });
        default:
            return state;
    }
}

// ===== Campaign Reducer ======

const initialCampaign = {
    name: "",
    bid: "",
};

function campaignReducer(state = initialCampaign, action) {
    switch (action.type) {
        case Actions.Upload.EDIT_CAMPAIGN_NAME:
            return update(state, { name: { $set: action.name } });
        case Actions.Upload.EDIT_CAMPAIGN_BID:
            return update(state, { bid: { $set: action.bid } });
        default:
            return state;
    }
}

// ===== Banner Reducer ======

const initialBanner = () => ({
    id: uniqueId(),
    titles: [],
    texts: [],
    hrefs: [],
    imageIds: [],
    href_params: {
        utm_source: "",
        utm_medium: "",
        utm_campaign: "",
        utm_term: "",
    },
});

function bannersReducer(state = [], action) {
    let bannerIndex = null;
    switch (action.type) {
        case Actions.Upload.EDIT_BANNER_TITLES:
        case Actions.Upload.EDIT_BANNER_TEXTS:
        case Actions.Upload.EDIT_BANNER_HREFS:
        case Actions.Upload.EDIT_BANNER_PARAMS:
        case Actions.Upload.SELECT_IMAGES:
            bannerIndex = state.findIndex((value) => value.id === action.bannerId);
            return update(state, {
                [bannerIndex]: {
                    $set: bannerReducer(state[bannerIndex], action),
                },
            });
        case Actions.Upload.ADD_BANNER:
            const newBanner = update(initialBanner(), {
                id: { $set: action.bannerId },
            });
            return update(state, { $push: [newBanner] });
        case Actions.Upload.COPY_BANNER:
            bannerIndex = state.findIndex((value) => value.id === action.fromBannerId);
            const copiedBanner = update(state[bannerIndex], {
                id: { $set: action.toBannerId },
            });
            return update(state, { $push: [copiedBanner] });
        case Actions.Upload.REMOVE_BANNER:
            bannerIndex = state.findIndex((value) => value.id === action.bannerId);
            return update(state, { $splice: [[bannerIndex, 1]] });
        default:
            return state;
    }
}

function bannerReducer(state = initialBanner(), action) {
    switch (action.type) {
        case Actions.Upload.EDIT_BANNER_TITLES:
            return update(state, { titles: { $set: action.titles } });
        case Actions.Upload.EDIT_BANNER_TEXTS:
            return update(state, { texts: { $set: action.texts } });
        case Actions.Upload.EDIT_BANNER_HREFS:
            return update(state, { hrefs: { $set: action.hrefs } });
        case Actions.Upload.EDIT_BANNER_PARAMS:
            return update(state, {
                href_params: { [action.paramKey]: { $set: action.paramValue } },
            });
        case Actions.Upload.SELECT_IMAGES:
            return update(state, { imageIds: { $set: action.imageIds } });
        default:
            return state;
    }
}

// ===== Geo Reducer ======

const initialGeo = {
    countryName: null,
    expandTypeName: null,
    regionIds: null,
};

function geoReducer(state = initialGeo, action) {
    switch (action.type) {
        case Actions.Upload.SELECT_COUNTRY:
            return update(state, { countryName: { $set: action.countryName } });
        case Actions.Upload.SELECT_EXPAND_TYPE:
            return update(state, {
                expandTypeName: { $set: action.expandTypeName },
            });
        case Actions.Upload.SELECT_REGIONS:
            return update(state, { regionIds: { $set: action.regionIds } });
        default:
            return state;
    }
}

const initialDirectLoader = {
    activeClientId: null,
    catalogPanel: undefined,
    clients: [],
    clientsStatus: getApiDraftStatus(),
    countries: [],
    countriesStatus: getApiDraftStatus(),
    initializationStatus: getApiDraftStatus(),
    previewPanel: undefined,
    window: WindowsEnum.LOADING,
};

function DirectLoaderReducer(state = initialDirectLoader, action) {
    const intermediateState = {
        previewPanel: previewPanelReducer(state.previewPanel, action),
        catalogPanel: catalogPanelReducer(state.catalogPanel, action),
        clients: state.clients.map((client) => clientReducer(client, action)),
        clientsStatus: state.clientsStatus,
        countries: state.countries,
        countriesStatus: state.countriesStatus,
        initializationStatus: state.initializationStatus,
        activeClientId: state.activeClientId,
        window: state.window,
    };

    switch (action.type) {
        case Actions.START_INITIALIZATION:
        case Actions.END_INITIALIZATION_SUCCESS:
        case Actions.END_INITIALIZATION_FAILED:
            return update(intermediateState, {
                initializationStatus: { $set: action.initializationStatus },
            });
        case Actions.VIEW_UPLOADING_PANEL:
            return update(intermediateState, {
                window: { $set: WindowsEnum.UPLOAD },
            });
        case Actions.VIEW_PREVIEW_PANEL:
            return update(intermediateState, {
                window: { $set: WindowsEnum.PREVIEW },
            });
        case Actions.VIEW_CATALOG_PANEL:
            return update(intermediateState, {
                window: { $set: WindowsEnum.CATALOG },
            });
        case Actions.VIEW_ERROR_PANEL:
            return update(intermediateState, {
                window: { $set: WindowsEnum.ERROR },
            });
        case Actions.VIEW_LOADING_PANEL:
            return update(intermediateState, {
                window: { $set: WindowsEnum.LOADING },
            });
        case Actions.SELECT_CLIENT:
            return update(intermediateState, {
                activeClientId: { $set: action.clientId },
            });
        case Actions.RECEIVE_REGIONS_FAILED:
        case Actions.REQUEST_REGIONS:
            return update(intermediateState, {
                countriesStatus: { $set: action.countriesStatus },
            });
        case Actions.RECEIVE_REGIONS_SUCCESS:
            return update(intermediateState, {
                countries: { $set: action.countries },
                countriesStatus: { $set: action.countriesStatus },
            });
        case Actions.RECEIVE_CLIENTS_FAILED:
        case Actions.REQUEST_CLIENTS:
            return update(intermediateState, {
                clientsStatus: { $set: action.clientsStatus },
            });
        case Actions.RECEIVE_CLIENTS_SUCCESS:
            const initialReceivedClient = (client) =>
                update(initialClient(), {
                    id: { $set: client.direct_id },
                    name: { $set: client.name },
                });
            const updatedClients = action.clients.map((client) => clientReducer(initialReceivedClient(client), action));
            return update(intermediateState, {
                clients: { $set: updatedClients },
                clientsStatus: { $set: action.clientsStatus },
            });
        default:
            return intermediateState;
    }
}

export default DirectLoaderReducer;
export {
    clientReducer,
    bannerReducer,
    adGroupReducer,
    imagesReducer,
    bannersReducer,
    adGroupsReducer,
    catalogPanelReducer,
    previewPanelReducer,
    geoReducer,
    campaignReducer,
    templatesReducer,
};
