import { some as _some, isEmpty } from "lodash";
import { createAction } from "redux-actions";
import { push } from "connected-react-router";
import qs from "query-string";
import { addErrorNotification } from "./utils/notifications/notificationsState";
import { getLocationSearchState } from "./segments/selectors";

export const changeLanguage = createAction("change language");

export const requestRoles = createAction("request roles");
export const receiveRoles = createAction("receive roles");
export const failedRoles = createAction("failed roles");
export const gotApi = createAction("got api");
export const failedApi = createAction("failed api");

export const updateNow = createAction("update time");

export function getApi(api) {
    return (dispatch) => {
        return api
            .then((api) => dispatch(gotApi({ host: api.url, version: api.spec.info.version })))
            .catch((error) => dispatch(failedApi(error)));
    };
}

export function fetchRoles(api) {
    return (dispatch) => {
        dispatch(requestRoles());

        return api
            .then((api) => api.apis.idm.getMyRoles)
            .then((method) => method())
            .then((response) => {
                let roles = response.obj.roles;
                return {
                    Admin: _some(roles, (role) => role.group === "lab" && role.subgroup === "admin"),
                    Siberia: _some(roles, (role) => role.group === "lab" && role.subgroup === "siberia"),
                    Geodirect: _some(roles, (role) => role.group === "distribution"),
                    Learning: _some(roles, (role) => role.group === "lab" && role.subgroup === "admin"),
                    ExpressionEditor: _some(
                        roles,
                        (role) => role.group === "lab" && role.subgroup === "expressions-editor"
                    ),
                    AudienceSegmentSharer: _some(
                        roles,
                        (role) => role.group === "lab" && role.subgroup === "audience-segments-sharer"
                    ),
                    // CRYPTA-15435: Meta role for CD
                    RestrictedSegments: _some(
                        roles,
                        (role) => role.group === "lab" && role.subgroup === "segments-not-allowed"
                    ),
                };
            })
            .then((access) => dispatch(receiveRoles(access)))
            .catch((error) => {
                if (error.obj) {
                    dispatch(failedRoles(error.obj.message));
                } else {
                    dispatch(failedRoles("Failed to fetch roles"));
                    dispatch(addErrorNotification(error, "Failed to fetch roles"));
                }
            });
    };
}

export const setEnvType = createAction("set env type");

export function fetchEnv() {
    return (dispatch) => {
        return fetch("/env.json")
            .then((response) => response.json())
            .then((env) => {
                const apiUrl = env.CLIENT_ENV_API;
                let envType;

                if (apiUrl.startsWith("https://api.")) {
                    envType = "production";
                } else {
                    envType = "testing";
                }

                dispatch(setEnvType(envType));
            })
            .catch((error) => {
                console.error("Unable to set envType");
            });
    };
}

export function pushToSearch(query) {
    return (dispatch) => {
        dispatch(push({ search: "?" + qs.stringify(query) }));
    };
}

export function pushQuery(queryObj) {
    return (dispatch, getState) => {
        let newQuery = qs.parse(getLocationSearchState(getState()));
        newQuery = { ...newQuery, ...queryObj };
        dispatch(pushToSearch(newQuery));
    };
}

export function removeQuery(queryKeys) {
    return (dispatch, getState) => {
        let newQuery = qs.parse(getLocationSearchState(getState()));

        queryKeys.forEach((key) => {
            if (!isEmpty(key) && key in newQuery) {
                delete newQuery[key];
            }
        });

        dispatch(pushToSearch(newQuery));
    };
}

export * from "audiences/actions";
export * from "segments/actions";
export * from "samples/actions";
