import Swagger from "swagger-client";
import { isUndefined, isEmpty } from "lodash";

function getEnv() {
    return fetch("/env.json", { credentials: "same-origin" }).then((response) => response.json());
}

function maybePublic(url) {
    // See CRYPTR-383
    if (window.location.hostname.startsWith("public.lab.")) {
        console.log("Using Public API");
        return url.replace("api.", "public.api.");
    } else {
        console.log("Using default API");
        return url;
    }
}

const HTTP_CODE_OLD_SESSION_ID = 419;
const HTTP_CODE_NOT_AUTHENTICATED = 401;

function createAPI(api_type) {
    return getEnv()
        .then((env) => {
            return new Swagger({
                url: maybePublic(env[api_type]),
                usePromise: true,
                requestInterceptor: (request) => {
                    request.credentials = "include";
                    return request;
                },
                // TODO: if everything is OK use the interceptor in other api clients
                responseInterceptor: (data) => {
                    if (data.status === HTTP_CODE_OLD_SESSION_ID) {
                        if (!isUndefined(data.obj) && !isEmpty(data.obj.authenticationUrl)) {
                            fetch(data.obj.authenticationUrl, {
                                credentials: "same-origin",
                            }).then((response) => {
                                if (response.ok) {
                                    console.info("You have been reauthenticated at yandex-team.ru");
                                }
                            });
                        }
                    } else if (data.status === HTTP_CODE_NOT_AUTHENTICATED) {
                        if (!isUndefined(data.obj) && !isEmpty(data.obj.authenticationUrl)) {
                            window.location.assign(data.obj.authenticationUrl);
                        }
                    }

                    return data;
                },
                success: function () {
                    console.log("Successfully connected to Crypta API");
                },
                error: function () {
                    alert("Failed to connect to API");
                },
            });
        })
        .then((api) => {
            return api;
        });
}

function createPhfAPI(api_type) {
    return getEnv()
        .then(
            (env) =>
                new Swagger({
                    url: env[api_type],
                    usePromise: true,
                    requestInterceptor: (request) => {
                        request.credentials = "include";
                        return request;
                    },
                })
        )
        .then((api) => {
            console.log("Create promise to api: ", api_type);
            return api.apis;
        });
}

export default class phfAPI {
    // Now methods of this class return objs as is. Better way is to transform in appropriate form before passing to ui
    // Can't find any type checks during SwaggerClient usage. Why we need this client at this case?

    constructor(api_type) {
        this.api = createPhfAPI(api_type);
    }

    getClients() {
        return this.api.then((api) => api.clients.get_client_list()).then((clients) => clients.obj);
    }

    getRegions() {
        return this.api.then((api) => api.regions.get_country_list()).then((regions) => regions.obj);
    }

    getTemplates(clientId) {
        return this.api
            .then((api) => api.clients.get_campaign_template_list({ client_id: clientId }))
            .then((templates) => templates.obj);
    }

    getTemplate(clientId, templateId) {
        return this.api
            .then((api) =>
                api.clients.get_template_controller({
                    client_id: clientId,
                    template_id: templateId,
                })
            )
            .then((template) => template.obj);
    }

    getCampaigns(clientId, templateId) {
        return this.api
            .then((api) =>
                api.clients.get_campaign_list({
                    client_id: clientId,
                    template_id: templateId,
                })
            )
            .then((campaigns) => campaigns.obj);
    }

    getAdGroups(clientId, templateId, campaignId) {
        return this.api
            .then((api) =>
                api.clients.get_ad_group_list({
                    client_id: clientId,
                    template_id: templateId,
                    campaign_id: campaignId,
                })
            )
            .then((adGroups) => adGroups.obj);
    }

    getBanners(clientId, templateId, campaignId, adGroupId) {
        return this.api
            .then((api) =>
                api.clients.get_banner_list({
                    client_id: clientId,
                    template_id: templateId,
                    campaign_id: campaignId,
                    ad_group_id: adGroupId,
                })
            )
            .then((banners) => banners.obj);
    }

    getPhrases(clientId, templateId, campaignId, adGroupId) {
        return this.api
            .then((api) =>
                api.clients.get_phrase_list({
                    client_id: clientId,
                    template_id: templateId,
                    campaign_id: campaignId,
                    ad_group_id: adGroupId,
                })
            )
            .then((phrases) => phrases.obj);
    }

    uploadImage(clientId, image) {
        return this.api
            .then((api) =>
                api.clients.post_image_upload({
                    client_id: clientId,
                    image: image,
                })
            )
            .then((result) => result.obj);
    }

    createTemplate(clientId, payload) {
        return this.api
            .then((api) =>
                api.clients.post_campaign_template_list({
                    client_id: clientId,
                    payload: payload,
                })
            )
            .then((template) => template.obj);
    }

    updateTemplate(clientId, templateId, payload) {
        return this.api
            .then((api) =>
                api.clients.put_template_controller({
                    template_id: templateId,
                    client_id: clientId,
                    payload: payload,
                })
            )
            .then((template) => template.obj);
    }

    uploadTemplate(clientId, templateId) {
        // Maybe this api call should return something
        // There is no checks if client is valid on server
        return this.api.then((api) =>
            api.clients.post_upload_template_to_direct({
                client_id: clientId,
                template_id: templateId,
            })
        );
    }

    moderateTemplate(clientId, templateId) {
        return this.api.then((api) =>
            api.clients.post_moderate_template_campaigns({
                client_id: clientId,
                template_id: templateId,
            })
        );
    }

    bidTemplate(clientId, templateId, payload) {
        return this.api.then((api) =>
            api.clients.post_bid_on_campaigns({
                client_id: clientId,
                template_id: templateId,
                payload: payload,
            })
        );
    }
}

const crypta_api = createAPI("CLIENT_ENV_API");
const phf_api = new phfAPI("CLIENT_ENV_PHF_API");
const auth_url = getEnv().then((env) => {
    return env["CLIENT_ENV_AUTH_URL"];
});

export { crypta_api, phf_api, auth_url };
