import React, { Component } from "react";
import { connect } from "react-redux";
import FontAwesome from "react-fontawesome";
import { Button } from "@lib/components/lego/button";

import { selectAudience, deleteAudience, deleteLabAudience, showModal } from "actions";
import { getSelectedAudienceIdState, getApiState } from "../selectors";
import State from "./State";
import Id from "../utils/Id";
import SimpleDate from "../utils/SimpleDate";
import { withTranslation } from "react-i18next";

@withTranslation("audience")
@connect(
    (state) => ({
        api: getApiState(state),
        selectedId: getSelectedAudienceIdState(state),
    }),
    (dispatch) => ({
        showModal: (modalName) => dispatch(showModal(modalName)),
        selectAudience: (audienceId) => dispatch(selectAudience(audienceId)),
        deleteAudience: (api, id) => dispatch(deleteAudience(api, id)),
        deleteLabAudience: (api, id) => dispatch(deleteLabAudience(api, id)),
    })
)
class AudiencesTableRow extends Component {
    handleLabDeletion() {
        const { api, audience, t } = this.props;
        const message = "[" + audience.id + "] " + t("titles.deleteFromLab") + "?";
        if (window.confirm(message)) {
            this.props.deleteLabAudience(api, audience.id);
        }
    }

    handleAudienceDeletion() {
        const { api, audience, t } = this.props;
        const message = "[" + audience.id + "] " + t("titles.deleteFromAudience") + "?";
        if (window.confirm(message)) {
            this.props.deleteAudience(api, audience.id);
        }
    }

    render() {
        const { audience, t, i18n } = this.props;
        const isSelected = this.props.selectedId && audience.id === this.props.selectedId;
        let lang = i18n.language;

        return (
            <tr
                className={`AudiencesTableRow ${isSelected ? "selected" : ""}`}
                onClick={() => this.props.selectAudience(audience.id)}
            >
                <td key="id">
                    {" "}
                    <Id copyable={true} value={audience.id} />
                </td>
                <td key="name">{audience.name}</td>
                <td key="login">
                    {" "}
                    <Id copyable={true} value={audience.login} />{" "}
                </td>
                <td key="externalId">
                    {" "}
                    <Id copyable={true} value={audience.externalId || "—"} />{" "}
                </td>
                <td key="state" className="AudienceState">
                    {" "}
                    <State state={audience.state} />{" "}
                </td>
                <td key="modified" className="SampleModified">
                    <SimpleDate lang={lang} value={audience.timestamps.modified} />
                </td>
                <td key="buttons" className="AudienceButtons">
                    <Button
                        view="clear"
                        theme="clear"
                        size="s"
                        title={t("titles.info")}
                        onClick={() => this.props.showModal("AudienceInfoModal")}
                    >
                        <FontAwesome name="info" />
                    </Button>
                    <Button
                        view="clear"
                        theme="clear"
                        size="s"
                        disabled={audience.state === "DELETED" || !audience.externalId}
                        title={t("titles.refresh")}
                        onClick={() => this.props.showModal("ModifyAudienceModal")}
                    >
                        <FontAwesome name="upload" />
                    </Button>
                    <Button
                        view="clear"
                        theme="clear"
                        size="s"
                        disabled={audience.state === "DELETED" || !audience.externalId}
                        title={t("titles.deleteFromAudience")}
                        onClick={() => this.handleAudienceDeletion()}
                    >
                        <FontAwesome name="user-times" />
                    </Button>
                    <Button
                        view="clear"
                        theme="clear"
                        size="s"
                        title={t("titles.deleteFromLab")}
                        onClick={() => this.handleLabDeletion()}
                    >
                        <FontAwesome name="trash" />
                    </Button>
                </td>
            </tr>
        );
    }
}

export default AudiencesTableRow;
