import React, { Component } from "react";
import { connect } from "react-redux";
import FontAwesome from "react-fontawesome";
import { Button, TextInput, Textarea, Spin, Select } from "@lib/components/lego";
import { pick as _pick, some as _some, isEmpty, isUndefined, debounce } from "lodash";

import { createAudience, createGrabId } from "actions";
import { getApiState, getAuthUrlState, getSrcTableColumnsState, getVisibleModalState } from "src/selectors";

import "./CreateAudienceModalBody.scss";
import { checkAuth, requestAuth } from "src/audiences/actions";
import { getAudiencesGlobalState } from "src/audiences/selectors";
import { withTranslation } from "react-i18next";
import ValidableYtPath from "../utils/ValidableYtPath";
import { validateTablePath } from "../utils/utilsActions";
import { Link } from "@yandex-lego/components/Link";

const AUDIENCE_LOGIN_PLACEHOLDER = "my-login";
const AUDIENCE_SOURCE_PATH_PLACEHOLDER = "//path/to/mySuperCoolAudience";

const TEXT_INPUT_SIZE = "m";

function sleep(ms) {
    return new Promise((resolve) => setTimeout(resolve, ms));
}

@withTranslation("audience")
@connect(
    (state) => ({
        api: getApiState(state),
        authUrl: getAuthUrlState(state),
        visible: getVisibleModalState(state, "CreateAudienceModal"),
        auth: getAudiencesGlobalState(state),
        srcYtColumns: getSrcTableColumnsState(state).columns,
        srcTableValid: !isEmpty(getSrcTableColumnsState(state).columns),
    }),
    (dispatch) => ({
        createAudience: (api, params, setState) => dispatch(createAudience(api, params, setState)),
        createGrabId: (api, extLogin, setState) => dispatch(createGrabId(api, extLogin, setState)),
        checkAuth: (api, extLogin, grabId, setState) => dispatch(checkAuth(api, extLogin, grabId, setState)),
        resetAuth: () => dispatch(requestAuth()),
        validateTablePath: (api, path) => dispatch(validateTablePath(api, path)),
    })
)
class CreateAudienceModalBody extends Component {
    constructor(props) {
        super(props);
        this.handleSourcePathChange = this.handleSourcePathChange.bind(this);

        this.state = this.getInitialState();
        this.validateTablePathThrottled = debounce(props.validateTablePath, 1000);
    }

    getInitialState() {
        return {
            name: "",
            purpose: "",
            login: "",
            grabId: "",
            authFinished: false,
            authAttempted: false,
            sourcePath: "",
            sourceField: "",

            error: null,
            loaded: true,
            buttonDisabled: true,
        };
    }

    componentDidUpdate(prevProps, prevState) {
        if (this.props.visible && !prevProps.visible) {
            this.setState(this.getInitialState());
        }

        const buttonDisabled = this.isButtonDisabled(this.getCreateAudienceParams(this.state));
        if (buttonDisabled !== prevState.buttonDisabled) {
            this.setState({ buttonDisabled: buttonDisabled });
        }

        if (this.state.login !== prevState.login) {
            this.props.resetAuth();
            this.setState({
                authFinished: false,
                authAttempted: false,
            });
        }

        if (prevProps.srcYtColumns !== this.props.srcYtColumns && this.props.srcYtColumns.length > 0) {
            this.setState({ sourceField: this.props.srcYtColumns[0].name });
        }
    }

    getCreateAudienceParams(obj) {
        return _pick(obj, ["name", "purpose", "login", "sourcePath", "sourceField", "grabId"]);
    }

    isButtonDisabled(params) {
        return _some(params, (each) => !each) || !this.props.auth.passportAuth;
    }

    async checkAuth(grabId) {
        let maxCount = 10;
        let count = 0;

        this.setState({ authAttempted: true, grabId: grabId });
        await this.props.checkAuth(this.props.api, this.state.login, grabId, this.setState.bind(this));

        while (this.props.auth.passportAuth !== true && count < maxCount) {
            await this.props.checkAuth(this.props.api, this.state.login, grabId, this.setState.bind(this));

            count++;
            await sleep(3000);
        }

        this.setState({ authFinished: true });
    }

    createAuth() {
        Promise.all([
            this.props.authUrl.then((authUrl) => authUrl),
            this.props
                .createGrabId(this.props.api, this.state.login, this.setState.bind(this))
                .then((grabId) => {
                    this.checkAuth(grabId);
                    return grabId;
                })
                .then((grabId) => {
                    let grabParams = new URLSearchParams();
                    grabParams.append("login", this.state.login);
                    grabParams.append("grab_id", grabId);

                    return grabParams;
                }),
        ]).then(([authUrl, grabParams]) => {
            window.open(authUrl + "?" + grabParams.toString(), "_blank");
        });
    }

    handleSourcePathChange(value) {
        const { api } = this.props;

        this.setState({ sourcePath: value.slice(0, 300) });
        this.validateTablePathThrottled(api, value);
    }

    getHint() {
        const { t, auth } = this.props;

        let hint = isEmpty(this.state.login) || this.state.authAttempted ? "" : t("create.validateHint");

        if (this.state.authAttempted && this.state.authFinished && !auth.passportAuth) {
            hint = (
                <span>
                    <span>{t("create.loginFailed")}</span>
                    <Link href="https://passport.yandex.ru" target="_blank">
                        passport.yandex.ru
                    </Link>
                </span>
            );
        }

        return hint;
    }

    render() {
        const { srcYtColumns, srcTableValid, t } = this.props;

        const loginValidationStatus = !this.state.authAttempted
            ? undefined
            : this.state.authFinished && !isUndefined(this.props.auth.passportAuth) && !this.props.auth.passportAuth
            ? "error"
            : undefined;

        const hintText = this.getHint();

        return (
            <div className="CreateAudienceModal-body">
                <table className="sample-params-table">
                    <tbody>
                        <tr>
                            <td className="sample-param-label">{t("create.name")}</td>
                            <td className="sample-param-value">
                                <TextInput
                                    view="default"
                                    size={TEXT_INPUT_SIZE}
                                    hasClear
                                    value={this.state.name}
                                    placeholder={t("create.placeholders.name")}
                                    onChange={(event) =>
                                        this.setState({
                                            name: event.target.value.slice(0, 100),
                                        })
                                    }
                                    disabled={!this.state.loaded}
                                />
                            </td>
                        </tr>
                        <tr>
                            <td className="sample-param-label">{t("create.purpose")}</td>
                            <td className="sample-param-value">
                                <Textarea
                                    view="default"
                                    size={TEXT_INPUT_SIZE}
                                    hasClear
                                    value={this.state.purpose}
                                    placeholder={t("create.placeholders.purpose")}
                                    onChange={(event) =>
                                        this.setState({
                                            purpose: event.target.value.slice(0, 1000),
                                        })
                                    }
                                    disabled={!this.state.loaded}
                                    rows={3}
                                />
                            </td>
                        </tr>
                        <tr>
                            <td className="sample-param-label">{t("create.login")}</td>
                            <td className="sample-param-value">
                                <span className="AudienceLoginInput-container">
                                    <span className="audience-login-input">
                                        <TextInput
                                            view="default"
                                            size={TEXT_INPUT_SIZE}
                                            state={loginValidationStatus}
                                            hint={hintText}
                                            hasClear
                                            value={this.state.login}
                                            placeholder={AUDIENCE_LOGIN_PLACEHOLDER}
                                            onChange={(event) =>
                                                this.setState({
                                                    login: event.target.value.slice(0, 50),
                                                })
                                            }
                                            disabled={!this.state.loaded}
                                        />
                                    </span>
                                    <span className="audience-login-domain">@yandex.ru</span>
                                    <span className="audience-validate-login-button">
                                        {this.state.authAttempted ? (
                                            !this.state.authFinished ? (
                                                <span>{t("create.checking")}</span>
                                            ) : this.props.auth.passportAuth ? (
                                                <FontAwesome name="check-circle" />
                                            ) : (
                                                <span />
                                            )
                                        ) : (
                                            <Button
                                                view="action"
                                                size="s"
                                                onClick={() => this.createAuth()}
                                                disabled={isEmpty(this.state.login) || isUndefined(this.state.login)}
                                            >
                                                {t("create.validate")}
                                            </Button>
                                        )}
                                    </span>
                                </span>
                            </td>
                        </tr>
                        <tr>
                            <td className="sample-param-label">{t("create.sourcePath")}</td>
                            <td className="sample-param-value">
                                <ValidableYtPath>
                                    <TextInput
                                        view="default"
                                        size={TEXT_INPUT_SIZE}
                                        hasClear
                                        value={this.state.sourcePath}
                                        placeholder={AUDIENCE_SOURCE_PATH_PLACEHOLDER}
                                        onChange={(event) => this.handleSourcePathChange(event.target.value)}
                                        disabled={!this.state.loaded}
                                    />
                                </ValidableYtPath>
                            </td>
                        </tr>
                        <tr>
                            <td className="sample-param-label">{t("create.sourceField")}</td>
                            <td className="sample-param-value">
                                <Select
                                    size={TEXT_INPUT_SIZE}
                                    view="default"
                                    width="max"
                                    onChange={(event) =>
                                        this.setState({ sourceField: event.target.value.slice(0, 100) })
                                    }
                                    value={this.state.sourceField}
                                    options={srcYtColumns
                                        // TODO(CRYPTA-15494) Support id types in API handle?
                                        .filter((column) => column.idTypes.includes("yandexuid"))
                                        .map((column) => ({
                                            value: column.name,
                                            content: column.name,
                                        }))}
                                    disabled={!this.state.loaded || srcYtColumns.length === 0}
                                />
                            </td>
                        </tr>
                    </tbody>
                </table>
                <Button
                    className="CreateAudienceModal-button"
                    view="action"
                    size="s"
                    onClick={() => {
                        this.setState({ loaded: false });
                        this.props.createAudience(
                            this.props.api,
                            this.getCreateAudienceParams(this.state),
                            this.setState.bind(this)
                        );
                    }}
                    disabled={!srcTableValid || this.state.buttonDisabled || !this.state.loaded}
                >
                    {t("create.create")}
                </Button>

                {!this.state.loaded && (
                    <div className="import-loading">
                        <Spin size="l" />
                    </div>
                )}
            </div>
        );
    }
}

export default CreateAudienceModalBody;
