import React, { useEffect, useState } from "react";
import { useTranslation } from "react-i18next";
import { connect, useDispatch } from "react-redux";
import { Button, Modal, Spin } from "@lib/components/lego";
import FontAwesome from "react-fontawesome";
import { reachGoal } from "metrika";
import { debounce, get as _get, isEmpty } from "lodash";

import LabModalHeader from "../../utils/LabModalHeader";
import ImportDatedSampleParams from "./DatedSampleParameters";

import { getSamplesState } from "../../samples/selectors";
import {
    getAccessState,
    getApiState,
    getLanguageState,
    getSrcTableColumnsState
} from "../../selectors";
import { resetYtTableColumns, validateTablePath, setActiveColumn } from "../../utils/utilsActions";
import { closeSampleCreateModal, fetchSamples, selectSample, changeSampleType } from "../../samples/actions";

import "../../samples/CreateSampleModal.scss";

const ACCESS_LEVEL_PLACEHOLDER = "SHARED";
const NAME = "name";
const PATH = "path";
const ID_TYPE = "idType";
const ID_KEY = "idKey";
const DATE_KEY = "dateKey";
const TTL = "ttl";
const ACCESS_LEVEL = "accessLevel";

const MIN_ROWS_NUMBER = 100;

const CreateDatedSampleModal = connect((state) => ({
    api: getApiState(state),
    srcYtColumns: getSrcTableColumnsState(state).columns,
    srcTableValid: !isEmpty(getSrcTableColumnsState(state).columns),
    visible: getSamplesState(state).modals.sampleCreate,
    isAdmin: getAccessState(state).Admin,
    lang: getLanguageState(state),
}))((props) => {
    const { api, srcYtColumns, visible, isAdmin, lang } = props;
    const { t } = useTranslation("createSample");
    const dispatch = useDispatch();

    const tooltipValues = {
        [PATH]: t("tooltip.source"),
        [ID_TYPE]: t("tooltip.idType"),
        [ID_KEY]: t("tooltip.idKey"),
        [DATE_KEY]: t("tooltip.dateKey"),
        [TTL]: t("tooltip.ttl"),
    };

    const [name, setName] = useState("");
    const [path, setPath] = useState("");
    const [idType, setIdType] = useState("");
    const [idKey, setIdKey] = useState("");
    const [dateKey, setDateKey] = useState("");
    const [ttl, setTtl] = useState("");
    const [accessLevel, setAccessLevel] = useState(ACCESS_LEVEL_PLACEHOLDER);

    const parametersDict = {
        [NAME]: setName,
        [PATH]: setPath,
        [ID_TYPE]: setIdType,
        [ID_KEY]: setIdKey,
        [DATE_KEY]: setDateKey,
        [TTL]: setTtl,
        [ACCESS_LEVEL]: setAccessLevel,
    };

    const [allFieldsDisabled, setAllFieldsDisabled] = useState(false);
    const [error, setError] = useState(null);
    const [loaded, setLoaded] = useState(true);
    const [response, setResponse] = useState(null);
    const [tooltip, setTooltip] = useState("");

    const validateTablePathThrottled = debounce((api, path) => dispatch(validateTablePath(api, path, MIN_ROWS_NUMBER)), 1000);

    const handleParameterChange = (value, type) => {
        if (Object.hasOwn(parametersDict, type)) {
            parametersDict[type](value);
            if (type === PATH) {
                validateTablePathThrottled(api, value);
            } else if (type === ID_KEY) {
                dispatch(setActiveColumn(value));
            }
        } else {
            console.log(parametersDict);
            console.log("Undefined type: ", type);
        }
    }

    const handleMouseOver = (value) => { setTooltip(tooltipValues[value]); };
    const handleMouseOut = () => { setTooltip(""); };

    useEffect(() => {
        if (srcYtColumns.length > 0) {
            setIdKey(srcYtColumns[0].name);
        }
    }, [srcYtColumns]);


    const createDatedSample = () => {
        setLoaded(false);
        setAllFieldsDisabled(true);
        setResponse(null);
        setError(null);

        dispatch(resetYtTableColumns());

        api.then((api) => api.apis.lab.createSample({
                [NAME]: name,
                [PATH]: path,
                [ID_TYPE]: idType,
                [ID_KEY]: idKey,
                [DATE_KEY]: dateKey,
                [TTL]: (ttl && Number(ttl) > 0) ? (Number(ttl) * 24 * 60 * 60) : null,
                [ACCESS_LEVEL]: accessLevel,
            }))
            .then((response) => {
                dispatch(reachGoal("createDatedSampleSuccess"));
                dispatch(closeSampleCreateModal());
                dispatch(selectSample(response.obj.id));
                dispatch(changeSampleType("my"));
                dispatch(
                    fetchSamples({
                        api: api,
                        lang: lang,
                        selected: response.obj.id,
                    })
                );
            })
            .catch((error) => {
                dispatch(reachGoal("createDatedSampleFail"));
                setError(error);
                setLoaded(true);
                setAllFieldsDisabled(false);
                setResponse(null);
            });
    }

    const handleHide = () => {
        dispatch(closeSampleCreateModal());
        dispatch(resetYtTableColumns());

        if (response) {
            dispatch(selectSample(response.id));
            dispatch(
                fetchSamples({
                    api: api,
                    lang: lang,
                    selected: response.id,
                })
            );
        }
    };

    const isCreateButtonDisabled = (!props.srcTableValid) || isEmpty(name) || isEmpty(idType) || isEmpty(dateKey) || allFieldsDisabled;

    return (
        <div className="CreateSample">
            <Modal theme="normal" visible={visible} onClose={handleHide}>
                <LabModalHeader
                    title={t("datedSampleCreation")}
                    hasClose
                    onClose={handleHide}
                />
                <div className="SampleCreationModal-body">
                    <div className="SampleCreationModal-tooltip">
                        <div className="SampleCreationModal-tooltip-text">
                            {tooltip && (
                                <div>
                                    <FontAwesome name="info-sign" /> {tooltip}
                                </div>
                            )}
                        </div>
                    </div>
                    <div className="SampleCreationModal-content">
                        <ImportDatedSampleParams
                            lang={lang}
                            onParameterChange={handleParameterChange}
                            name={name}
                            path={path}
                            idType={idType}
                            idKey={idKey}
                            dateKey={dateKey}
                            ttl={ttl}
                            accessLevel={accessLevel}
                            disabled={allFieldsDisabled}
                            isAdmin={isAdmin}
                            onMouseOver={handleMouseOver}
                            onMouseOut={handleMouseOut}
                        />
                    </div>
                    {!loaded && (
                        <div className="import-loading">
                            <Spin progress size="l" />
                        </div>
                    )}
                    {response && (
                        <div className="import-ok-message">
                            {t("ready")}
                            <div className="sample-id-imported" onClick={handleHide}>
                                {response.id}
                            </div>
                        </div>
                    )}
                    {error && (
                        <div className="import-error-message">
                            {t("error")} ({_get(error, "obj.code", "")})
                        </div>
                    )}
                    <div className="SubmitButton">
                        <Button
                            view="action"
                            theme="action"
                            size="s"
                            onClick={createDatedSample}
                            disabled={isCreateButtonDisabled}
                        >
                            {t("create")}
                        </Button>
                    </div>
                </div>
            </Modal>
        </div>
    );
});


export default CreateDatedSampleModal;
