import React, { useEffect, useState } from "react";
import { connect, useDispatch } from "react-redux";
import { useTranslation } from "react-i18next";
import { Button, Modal, TextInput } from "@lib/components/lego";
import Loading from "@lib/components/loading";

import Failable from "../../utils/Failable";
import DatedSampleViewsTable from './ViewsTable';
import SampleViewTableModal from "../../samples/SampleViewTableModal";
import LabModalHeader from "../../utils/LabModalHeader";

import { getApiState, getSamplesState } from "../../selectors";
import { closeSampleViews, fetchViews, setCreateSampleViewFormVisibility } from "../../samples/actions";
import { createView } from "../../samples/actions";

import "../../samples/SampleViewsModal.scss";


const CreateLookalikeSection = connect((state) => ({
    api: getApiState(state),
    waiting: getSamplesState(state).creatingView,
    error: getSamplesState(state).creatingViewErrorCode !== null,
    errorCode: getSamplesState(state).creatingViewErrorCode,
}))((props) => {
    const { sample, api, waiting, error, errorCode } = props;
    const { t } = useTranslation(["samples"]);

    const [lookalikeCount, setLookalikeCount] = useState(1000);

    const dispatch = useDispatch();

    const submitCreateLookalikeView = () => {
        dispatch(
            createView(api, sample, {
                id: sample.id,
                type: "lookalike",
                count: lookalikeCount,
                useDates: "true",
            })
        );
    };

    const hideForm = () => {
        dispatch(setCreateSampleViewFormVisibility(false));
    };

    const commonFailedMessage = t("views.failed");

    const errorLabels = {
        EQUAL_ID_TYPE: t("views.equalIdType"),
        CANNOT_CREATE_VIEW_FOR_THAT_SAMPLE: t("views.cantCreate"),
        MISSED_ROLE_FOR_SUCH_VIEW: t("views.noRole"),
        SAMPLE_IS_BROKEN: t("views.broken"),

        [undefined]: commonFailedMessage,
    };

    const errorMessage = errorLabels[errorCode] ? t("views.failed") + ". " + errorLabels[errorCode] : t("views.failed");

    const renderLookalikeParameters = () => {
        return (
            <div>
                <span className="CreateView-title">{t("views.type.lookalike")}</span>
                <div className="CreateView-lookalike">
                    <table className="CreateView-parameters">
                        <tbody>
                            <tr>
                                <td className="CreateView-parameter-label">{t("views.lookalikeCount")}</td>
                                <td className="CreateView-parameter-value">
                                    <TextInput
                                        view="default"
                                        theme="normal"
                                        size="m"
                                        hasClear
                                        type="number"
                                        value={lookalikeCount}
                                        placeholder="1000"
                                        onChange={(event) => setLookalikeCount(event.target.value)}
                                    />
                                </td>
                            </tr>
                        </tbody>
                    </table>
                </div>
            </div>
        );
    };

    return (
        <div className="CreateViewSection-container">
            <div className="CreateViewSection-parameters">
                {renderLookalikeParameters()}
            </div>
            <div className="CreateView-button">
                <Button view="action" size="m" onClick={submitCreateLookalikeView} disabled={waiting}>
                    {t("views.add")}
                </Button>
                <Button view="pseudo" size="m" onClick={hideForm}>
                    {t("views.hideForm")}
                </Button>
            </div>
            {error && <div className="CreateView-status">{errorMessage}</div>}
        </div>
    );
});


const DatedSampleViewsModal  = connect(state => ({
    api: getApiState(state),
    visible: getSamplesState(state).modals.views,
    createViewFormVisible: getSamplesState(state).createSampleViewFormVisible,
    loaded: getSamplesState(state).loadedViews,
    failed: getSamplesState(state).failedViews,
    views: getSamplesState(state).views,
}))((props) => {
    const { sample, api, visible, createViewFormVisible, loaded, failed, views } = props;
    const { t } = useTranslation(["samples"]);
    const dispatch = useDispatch();

    const fetchViews_ = () => dispatch(fetchViews(api, sample));

    useEffect(() => {
        if (sample) {
            fetchViews_();
        }
    }, [sample]);

    return (
        <div className="SampleViewsModal-container">
            <Modal
                view="default"
                theme="normal"
                visible={visible}
                zIndex={1}
                onClose={() => dispatch(closeSampleViews())}
            >
                {visible && (
                    <div className="SampleViews">
                        <LabModalHeader
                            title={`${t("viewsOf")} ${sample.name}`}
                            hasClose
                            onClose={() => dispatch(closeSampleViews())}
                        />

                        <Failable failed={failed} retry={fetchViews_}>
                            <Loading loaded={loaded}>
                                <DatedSampleViewsTable views={views} />

                                {createViewFormVisible ? (
                                    <CreateLookalikeSection sample={sample} />
                                ) : (
                                    <div className="SampleViewsModal-show-create-view-section-button">
                                        <Button
                                            view="action"
                                            theme="action"
                                            size="m"
                                            onClick={() => dispatch(setCreateSampleViewFormVisibility(true))}
                                        >
                                            {t("views.createLookalike")}
                                        </Button>
                                    </div>
                                )}
                            </Loading>
                        </Failable>
                    </div>
                )}
            </Modal>

            <SampleViewTableModal />
        </div>
    );
});


export default DatedSampleViewsModal;
