import React, { Component } from "react";
import { debounce, isEmpty, get as _get } from "lodash";
import { Button, Modal, TextInput, Spin, Select } from "@lib/components/lego";
import FontAwesome from "react-fontawesome";
import IdTypeSelect from "./IdTypeSelect";
import AccessLevelSelect from "./AccessLevelSelect";
import helpPic from "../images/help.svg";
import { connect } from "react-redux";
import {
    getLanguageState,
    getApiState,
    getSamplesState,
    getSelectedSampleIdState,
    getAccessState,
    getSrcTableColumnsState,
} from "../selectors";
import { closeSampleCreateModal, createSample, selectSample, fetchSamples } from "./actions";
import ValidableYtPath from "../utils/ValidableYtPath";
import { resetYtTableColumns, setActiveColumn, validateTablePath } from "../utils/utilsActions";
import Link from "@lib/components/link";
import { withTranslation } from "react-i18next";
import LabModalHeader from "../utils/LabModalHeader";

import "./CreateSampleModal.scss";

const LABORATORY_SAMPLES_HELP_URL = "https://wiki.yandex-team.ru/crypta/crypta-up/lab/samples/#sozdanievyborki";

const YT_PATH_PLACEHOLDER = "//path/to/mySuperCoolSample";
const ID_TYPE_PLACEHODLER = "yandexuid";
const ACCESS_LEVEL_PLACEHOLDER = "SHARED";
const TTL_PLACEHOLDER = "7";

const TEXT_INPUT_SIZE = "m";

const bigSpinner = (
    <div className="import-loading">
        <Spin progress size="l" />
    </div>
);

@withTranslation("createSample")
@connect(
    (state) => ({
        api: getApiState(state),
        srcTableValid: !isEmpty(getSrcTableColumnsState(state).columns),
        srcYtColumns: getSrcTableColumnsState(state).columns,
    }),
    (dispatch) => ({
        validateTablePath: (api, path) => dispatch(validateTablePath(api, path)),
        resetYtTableColumns: () => dispatch(resetYtTableColumns()),
    })
)
class CreateSampleModal extends Component {
    constructor(props) {
        super(props);
        this.handleChangePath = this.handleChangePath.bind(this);
        this.handleChangeSampleName = this.handleChangeSampleName.bind(this);
        this.handleChangeIdType = this.handleChangeIdType.bind(this);
        this.handleChangeIdKey = this.handleChangeIdKey.bind(this);
        this.handleChangeGroupingKey = this.handleChangeGroupingKey.bind(this);
        this.handleChangeTtl = this.handleChangeTtl.bind(this);
        this.handleChangeAccessLevel = this.handleChangeAccessLevel.bind(this);
        this.handleHide = this.handleHide.bind(this);
        this.handleMouseOver = this.handleMouseOver.bind(this);
        this.handleMouseOut = this.handleMouseOut.bind(this);

        this.validateTablePathThrottled = debounce(props.validateTablePath, 1000);

        this.state = {
            path: "",
            name: "",
            idType: ID_TYPE_PLACEHODLER,
            idKey: "",
            ttl: "",
            groupingKey: "",
            accessLevel: ACCESS_LEVEL_PLACEHOLDER,
            allFieldsDisabled: false,
            tooltip: "",

            error: null,
            response: null,
            loaded: true,
        };
    }

    componentDidMount() {
        const { t } = this.props;
        this.tooltip = {
            Source: t("tooltip.source"),
            IdType: t("tooltip.idType"),
            IdKey: t("tooltip.idKey"),
            GroupedBy: t("tooltip.groupedBy"),
            Ttl: t("tooltip.ttl"),
        };
    }

    componentDidUpdate(prevProps) {
        if (prevProps.srcYtColumns !== this.props.srcYtColumns && this.props.srcYtColumns.length > 0) {
            this.setState({ idKey: this.props.srcYtColumns[0].name });
        }
    }

    handleChangePath(value) {
        const { api } = this.props;

        this.setState({ path: value });
        this.validateTablePathThrottled(api, value);
    }

    handleChangeSampleName(value) {
        this.setState({ name: value });
    }

    handleChangeIdType(value) {
        this.setState({ idType: value });
    }

    handleChangeIdKey(value) {
        this.setState({ idKey: value });
        this.props.dispatch(setActiveColumn(value));
    }

    handleChangeGroupingKey(value) {
        this.setState({ groupingKey: value });
    }

    handleChangeTtl(value) {
        this.setState({ ttl: value });
    }

    handleChangeAccessLevel(value) {
        this.setState({ accessLevel: value });
    }

    handleHide() {
        this.setState(
            {
                path: "",
                name: "",
                idType: ID_TYPE_PLACEHODLER,
                idKey: "",
                ttl: "",
                groupingKey: "",
                accessLevel: ACCESS_LEVEL_PLACEHOLDER,
                allFieldsDisabled: false,

                error: null,
                response: null,
                loaded: true,
            },
            () => {
                this.props.dispatch(closeSampleCreateModal());
                this.props.dispatch(resetYtTableColumns());

                if (this.state.response) {
                    this.props.dispatch(selectSample(this.state.response.id));
                    this.props.dispatch(
                        fetchSamples({
                            api: this.props.api,
                            lang: this.props.lang,
                            selected: this.state.response.id,
                        })
                    );
                }
            }
        );
    }

    handleMouseOver(value) {
        this.setState({
            tooltip: this.tooltip[value],
        });
    }

    handleMouseOut() {
        this.setState({ tooltip: "" });
    }

    render() {
        const { visible, lang, isAdmin, srcTableValid, t } = this.props;

        return (
            <div className="import-sample-modal">
                <Modal
                    theme="normal"
                    visible={visible}
                    onClose={() => {
                        this.props.resetYtTableColumns();
                        this.props.dispatch(closeSampleCreateModal());
                    }}
                >
                    <LabModalHeader
                        title={t("creation")}
                        info={
                            <Link
                                className="SampleCreationModal-help-link"
                                href={LABORATORY_SAMPLES_HELP_URL}
                                title={t("userGuide")}
                                target="_blank"
                            >
                                <img alt="" className="SampleCreationModal-help-pic" src={helpPic} />
                            </Link>
                        }
                        hasClose
                        onClose={() => {
                            this.props.resetYtTableColumns();
                            this.props.dispatch(closeSampleCreateModal());
                        }}
                    />
                    <div className="SampleCreationModal">
                        <div className="SampleCreationModal-tooltip">
                            <div className="SampleCreationModal-tooltip-text">
                                {this.state.tooltip && (
                                    <div>
                                        <FontAwesome name="info-sign" /> {this.state.tooltip}
                                    </div>
                                )}
                            </div>
                        </div>
                        <div className="SampleCreationModal-body">
                            <div className="sample-modal-content">
                                <ImportSampleParams
                                    lang={lang}
                                    path={this.state.path}
                                    onPathChange={(path) => this.handleChangePath(path)}
                                    name={this.state.name}
                                    onNameChange={(name) => this.handleChangeSampleName(name)}
                                    idType={this.state.idType}
                                    onIdTypeChange={this.handleChangeIdType}
                                    idKey={this.state.idKey}
                                    onIdKeyChange={(idKey) => this.handleChangeIdKey(idKey)}
                                    groupingKey={this.state.groupingKey}
                                    onGroupingKeyChange={(groupingKey) => this.handleChangeGroupingKey(groupingKey)}
                                    ttl={this.state.ttl}
                                    onTtlChange={(ttl) => this.handleChangeTtl(ttl)}
                                    accessLevel={this.state.accessLevel}
                                    onAccessLevelChange={(accessLevel) => this.handleChangeAccessLevel(accessLevel)}
                                    disabled={this.state.allFieldsDisabled}
                                    isAdmin={isAdmin}
                                    onMouseOver={this.handleMouseOver}
                                    onMouseOut={this.handleMouseOut}
                                />
                            </div>

                            {!this.state.loaded && bigSpinner}

                            {this.state.response && (
                                <div className="import-ok-message">
                                    {t("ready")}
                                    <div className="sample-id-imported" onClick={() => this.handleHide()}>
                                        {this.state.response.id}
                                    </div>
                                </div>
                            )}

                            {this.state.error && (
                                <div className="import-error-message">
                                    {t("error")} ({_get(this.state.error, "obj.code", "")})
                                </div>
                            )}

                            <div
                                style={{
                                    position: "absolute",
                                    bottom: "5%",
                                    right: "5%",
                                }}
                            >
                                <Button
                                    view="action"
                                    theme="action"
                                    size="s"
                                    onClick={() =>
                                        this.props.dispatch(
                                            createSample(
                                                this.props.api,
                                                this.props.lang,
                                                (x, u) => this.setState(x, u),
                                                this.state
                                            )
                                        )
                                    }
                                    disabled={!srcTableValid || !this.state.name || this.state.allFieldsDisabled}
                                >
                                    {t("create")}
                                </Button>
                            </div>
                        </div>
                    </div>
                </Modal>
            </div>
        );
    }
}

@withTranslation("createSample")
@connect((state) => ({
    srcYtColumns: getSrcTableColumnsState(state).columns,
    srcTableValid: !isEmpty(getSrcTableColumnsState(state).columns),
}))
class ImportSampleParams extends Component {
    render() {
        const {
            path,
            name,
            idType,
            idKey,
            groupingKey,
            ttl,
            accessLevel,
            disabled,
            isAdmin,
            onMouseOver,
            onMouseOut,
            srcYtColumns,
            srcTableValid,
            t,
        } = this.props;

        return (
            <div className="import-sample-params">
                <table className="sample-params-table">
                    <tbody>
                        <tr>
                            <td className="sample-param-label">{t("name")}</td>
                            <td className="sample-param-value">
                                <TextInput
                                    view="default"
                                    theme="normal"
                                    size={TEXT_INPUT_SIZE}
                                    hasClear
                                    value={name}
                                    placeholder={t("namePlaceholder")}
                                    onChange={(event) => this.props.onNameChange(event.target.value)}
                                    disabled={disabled}
                                />
                            </td>
                        </tr>
                        <tr onMouseOver={() => onMouseOver("Source")} onMouseOut={onMouseOut}>
                            <td className="sample-param-label">{t("source")}</td>
                            <td className="sample-param-value">
                                <ValidableYtPath>
                                    <TextInput
                                        view="default"
                                        theme="normal"
                                        size={TEXT_INPUT_SIZE}
                                        hasClear
                                        value={path}
                                        placeholder={YT_PATH_PLACEHOLDER}
                                        onChange={(event) => this.props.onPathChange(event.target.value)}
                                        disabled={disabled}
                                    />
                                </ValidableYtPath>
                            </td>
                        </tr>
                        <tr>
                            <td className="sample-param-label">{t("column")}</td>
                            <td className="sample-param-value">
                                <Select
                                    size={TEXT_INPUT_SIZE}
                                    view="default"
                                    width="max"
                                    onChange={(event) => this.props.onIdKeyChange(event.target.value)}
                                    value={idKey}
                                    options={srcYtColumns.map((column) => ({
                                        value: column.name,
                                        content: column.name,
                                    }))}
                                    disabled={disabled || !srcTableValid}
                                />
                            </td>
                        </tr>
                        <tr>
                            <td className="sample-param-label">
                                <span
                                    className="id-select"
                                    onMouseOver={() => onMouseOver("IdType")}
                                    onMouseOut={onMouseOut}
                                >
                                    {t("idType")}
                                </span>
                            </td>
                            <td className="sample-param-value">
                                <IdTypeSelect value={idType} onChange={this.props.onIdTypeChange} />
                            </td>
                        </tr>
                        <tr>
                            <td className="sample-param-label">{t("groupedByField")}</td>
                            <td className="sample-param-value">
                                <span onMouseOver={() => onMouseOver("GroupedBy")} onMouseOut={onMouseOut}>
                                    <TextInput
                                        view="default"
                                        theme="normal"
                                        size={TEXT_INPUT_SIZE}
                                        hasClear
                                        value={groupingKey}
                                        placeholder={t("withoutGrouping")}
                                        onChange={(event) => this.props.onGroupingKeyChange(event.target.value)}
                                        disabled={disabled}
                                    />
                                </span>
                            </td>
                        </tr>
                        <tr>
                            <td className="sample-param-label">
                                {t("storeFor")} ({t("days")})
                            </td>
                            <td className="sample-param-value">
                                <span onMouseOver={() => onMouseOver("Ttl")} onMouseOut={onMouseOut}>
                                    <span className="ttl-input">
                                        <TextInput
                                            view="default"
                                            theme="normal"
                                            size={TEXT_INPUT_SIZE}
                                            value={ttl}
                                            placeholder={TTL_PLACEHOLDER}
                                            onChange={(event) => this.props.onTtlChange(event.target.value)}
                                            disabled={disabled}
                                        />
                                    </span>
                                </span>
                            </td>
                        </tr>
                        <tr>
                            <td className="sample-param-label">{t("access")}</td>
                            <td className="sample-param-value">
                                <AccessLevelSelect
                                    value={accessLevel}
                                    onChange={this.props.onAccessLevelChange}
                                    isAdmin={isAdmin}
                                />
                            </td>
                        </tr>
                    </tbody>
                </table>
            </div>
        );
    }
}

const mapStateToProps = (state) => {
    return {
        api: getApiState(state),
        lang: getLanguageState(state),
        selected: getSelectedSampleIdState(state),
        visible: getSamplesState(state).modals.sampleCreate,
        isAdmin: getAccessState(state).Admin,
    };
};

export default connect(mapStateToProps)(CreateSampleModal);
