import React, { useState } from "react";
import { useTranslation } from "react-i18next";
import { useDispatch, connect } from "react-redux";
import { Button, RadioButton, Select, TextInput } from "@lib/components/lego";
import { createView, setCreateSampleViewFormVisibility } from "./actions";
import { getApiState, getSamplesState } from "../selectors";

import "./CreateViewSection.scss";

const INITIAL_VIEW_TYPE = "matching";

const CreateViewSection = connect((state) => ({
    api: getApiState(state),
    waiting: getSamplesState(state).creatingView,
    error: getSamplesState(state).creatingViewErrorCode !== null,
    errorCode: getSamplesState(state).creatingViewErrorCode,
}))((props) => {
    const { sample, api, waiting, error, errorCode } = props;

    const { t } = useTranslation("samples");

    const [viewType, setViewType] = useState(INITIAL_VIEW_TYPE);
    const [idType, setIdType] = useState("LAB_ID_YANDEXUID");
    const [hashingMethod, setHashingMethod] = useState("HM_IDENTITY");
    const [includeOriginal, setIncludeOriginal] = useState("false");
    const [matchingScope, setMatchingScope] = useState("CROSS_DEVICE");
    const [lookalikeCount, setLookalikeCount] = useState(1000);

    const dispatch = useDispatch();

    const submitCreateView = () => {
        dispatch(
            createView(api, sample, {
                id: sample.id,
                hashingMethod: hashingMethod,
                identifier: idType,
                includeOriginal: includeOriginal,
                scope: matchingScope,
                type: viewType,
                count: lookalikeCount,
                useDates: "false",
            })
        );
    };

    const hideForm = () => {
        dispatch(setCreateSampleViewFormVisibility(false));
    };

    const idTypes = {
        LAB_ID_YANDEXUID: "yandexuid",
        LAB_ID_IDFA_GAID: "IDFA/GAID",
        LAB_ID_MM_DEVICE_ID: "AppMetrica DeviceID",
        LAB_ID_UUID: "UUID",
        LAB_ID_EMAIL: "email",
        LAB_ID_PHONE: t("views.phone"),
        LAB_ID_LOGIN: t("views.login"),
        LAB_ID_PUID: "puid",
        LAB_ID_CRYPTA_ID: "CryptaID",
        LAB_ID_UNKNOWN: "Неизвестно",
    };

    const hashingMethods = {
        HM_IDENTITY: t("views.withoutHashing"),
        HM_MD5: "MD5",
        HM_SHA256: "SHA-256",
    };

    const includeOriginals = {
        true: t("views.yes"),
        false: t("views.no"),
    };

    const matchingScopes = {
        CROSS_DEVICE: "cross-device",
        IN_DEVICE: "in-device",
    };

    const commonFailedMessage = t("views.failed");

    const errorLabels = {
        EQUAL_ID_TYPE: t("views.equalIdType"),
        CANNOT_CREATE_VIEW_FOR_THAT_SAMPLE: t("views.cantCreate"),
        MISSED_ROLE_FOR_SUCH_VIEW: t("views.noRole"),
        SAMPLE_IS_BROKEN: t("views.broken"),

        [undefined]: commonFailedMessage,
    };

    const errorMessage = errorLabels[errorCode] ? t("views.failed") + ". " + errorLabels[errorCode] : t("views.failed");

    const renderMatchingParameters = () => {
        // TODO: collapse this somehow pretty

        return (
            <div className="CreateView-matching">
                <table className="CreateView-parameters">
                    <tbody>
                        <tr>
                            <td className="CreateView-parameter-label">{t("views.idTypeLabel")}</td>
                            <td className="CreateView-parameter-value">
                                <Select
                                    view="default"
                                    size="s"
                                    width="fixed"
                                    value={idType}
                                    onChange={(event) => setIdType(event.target.value)}
                                    options={Object.keys(idTypes).map((item) => ({
                                        value: item,
                                        content: idTypes[item],
                                    }))}
                                />
                            </td>
                        </tr>
                        <tr>
                            <td className="CreateView-parameter-label">{t("views.hashingMethodLabel")}</td>
                            <td className="CreateView-parameter-value">
                                <RadioButton
                                    view="default"
                                    size="s"
                                    value={hashingMethod}
                                    onChange={(event) => setHashingMethod(event.target.value)}
                                    options={Object.keys(hashingMethods).map((item) => ({
                                        value: item,
                                        children: hashingMethods[item],
                                    }))}
                                />
                            </td>
                        </tr>
                        <tr>
                            <td className="CreateView-parameter-label">{t("views.includeOriginal")}</td>
                            <td className="CreateView-parameter-value">
                                <RadioButton
                                    view="default"
                                    size="s"
                                    value={includeOriginal}
                                    onChange={(event) => setIncludeOriginal(event.target.value)}
                                    options={Object.keys(includeOriginals).map((item) => ({
                                        value: item,
                                        children: includeOriginals[item],
                                    }))}
                                />
                            </td>
                        </tr>
                        <tr>
                            <td className="CreateView-parameter-label">{t("views.matchingScopeLabel")}</td>
                            <td className="CreateView-parameter-value">
                                <RadioButton
                                    view="default"
                                    size="s"
                                    width="fixed"
                                    value={matchingScope}
                                    onChange={(event) => setMatchingScope(event.target.value)}
                                    options={Object.keys(matchingScopes).map((item) => ({
                                        value: item,
                                        children: matchingScopes[item],
                                    }))}
                                />
                            </td>
                        </tr>
                    </tbody>
                </table>
            </div>
        );
    };

    const renderCryptaIdStatisticsParameters = () => {
        return <div className="CreateView-cryptaid">TBD</div>;
    };

    const renderLookalikeParameters = () => {
        return (
            <div className="CreateView-lookalike">
                <span className="CreateView-lookalike-label">{t("views.lookalikeCount")}</span>
                <span className="CreateView-lookalike-value">
                    <TextInput
                        view="default"
                        theme="normal"
                        size="m"
                        hasClear
                        type="number"
                        value={lookalikeCount}
                        placeholder="1000"
                        onChange={(event) => setLookalikeCount(event.target.value)}
                    />
                </span>
            </div>
        );
    };

    return (
        <div className="CreateViewSection-container">
            <div className="CreateViewSection-parameters">
                <div className="CreateViewSection-type-select">
                    <RadioButton
                        value={viewType}
                        view="default"
                        size="s"
                        onChange={(event) => setViewType(event.target.value)}
                        options={[
                            { value: INITIAL_VIEW_TYPE, children: t("views.type.matching") },
                            { value: "crypta_id_statistics", children: t("views.type.cryptaIdStatistics") },
                            { value: "lookalike", children: t("views.type.lookalike") },
                        ]}
                    ></RadioButton>
                </div>

                <div className="CreateViewSection-parameters-values">
                    {viewType === INITIAL_VIEW_TYPE
                        ? renderMatchingParameters()
                        : viewType === "crypta_id_statistics"
                        ? renderCryptaIdStatisticsParameters()
                        : viewType === "lookalike"
                        ? renderLookalikeParameters()
                        : null}
                </div>
            </div>

            <div className="CreateView-button">
                <Button view="action" size="m" onClick={submitCreateView} disabled={waiting}>
                    {t("views.add")}
                </Button>

                <Button view="pseudo" size="m" onClick={hideForm}>
                    {t("views.hideForm")}
                </Button>
            </div>

            {error && <div className="CreateView-status">{errorMessage}</div>}
        </div>
    );
});

export default CreateViewSection;
