import React, { useEffect } from "react";
import { connect, useDispatch } from "react-redux";
import { useTranslation } from "react-i18next";
import { Button, Modal } from "@lib/components/lego";
import FontAwesome from "react-fontawesome";
import Link from "@lib/components/link";
import Loading from "@lib/components/loading";
import Id from "../utils/Id";
import YtPathLink from "../utils/YtPath";
import { map as _map } from "lodash";
import Failable from "../utils/Failable";
import { getApiState, getSamplesState } from "../selectors";
import { closeSampleViews, fetchViews, setCreateSampleViewFormVisibility } from "./actions";
import helpPic from "../images/help.svg";

import { openSampleTablePreview } from "src/samples/actions";
import SampleViewTableModal from "src/samples/SampleViewTableModal";
import CreateViewSection from "./CreateViewSection";
import LabModalHeader from "../utils/LabModalHeader";

import "./SampleViewsModal.scss";

const MATCHING_WIKI_URL = "https://wiki.yandex-team.ru/crypta/crypta-up/lab/Matching/";

const SampleViewsRow = connect((state) => ({
    api: getApiState(state),
}))((props) => {
    const { view, api } = props;
    const { Matching: matching, Lookalike: lookalike } = view.Options;
    const { SampleID: sampleId, ID: id, Path: tablePath } = view;

    const dispatch = useDispatch();
    const { t } = useTranslation("sampleView");

    const openPreview = (api, sampleId, viewId) => dispatch(openSampleTablePreview(api, sampleId, viewId));

    const getType = () => {
        if (view.Type === "MATCHING") {
            let type = t("idType." + matching.IdType);

            if (matching.HashingMethod !== "HM_IDENTITY") type += ", " + t("hashingMethod." + matching.HashingMethod);

            return type;
        } else if (view.Type === "LOOKALIKE") {
            let type = "yandexuid";
            if (lookalike.UseDates) type = "identity";
            return type + ", " + lookalike.Counts.Output;
        } else if (view.Type === "CRYPTA_ID_STATISTICS") {
            return matching.Key;
        } else {
            return "";
        }
    };

    const getIncludeOriginal = () => {
        if (view.Options.Matching && view.Options.Matching.IncludeOriginal) {
            return <FontAwesome name="check" />;
        } else {
            return <FontAwesome style={{ color: "lightgrey" }} name="times" />;
        }
    };

    const scope = view.Options.Matching ? t("matchingScope." + view.Options.Matching.Scope) : "?";

    const renderTable = () => {
        return (
            <div className="SampleTablePath">
                <span className="SamplePreview">
                    <Button
                        view="clear"
                        theme="normal"
                        size="s"
                        title={t("instantView")}
                        onClick={() => openPreview(api, sampleId, id)}
                    >
                        <FontAwesome name="search" />
                    </Button>
                </span>

                <YtPathLink tablePath={tablePath} linkTitle={t("toToYt")} />
            </div>
        );
    };

    return (
        <tr className="SamplesViewsTableRow">
            <td>
                <Id value={view.ID} copyable={false} />
            </td>
            <td>{t("description." + view.Type)}</td>
            <td>{getType()}</td>
            <td>{getIncludeOriginal()}</td>
            <td>{scope}</td>
            <td>{renderTable()}</td>
            <td>{t("state." + view.State)}</td>
        </tr>
    );
});

const SampleViewsTable = (props) => {
    const views = props.views;
    const { t } = useTranslation("samples");

    return (
        <table className="SamplesViewsTable">
            <thead key="head">
                <tr className="SamplesViewsTableHeader">
                    <th>{t("viewHeaders.id")}</th>
                    <th>{t("viewHeaders.description")}</th>
                    <th>{t("viewHeaders.idType")}</th>
                    <th>{t("viewHeaders.includeOriginal")}</th>
                    <th>{t("viewHeaders.scope")}</th>
                    <th>{t("viewHeaders.path")}</th>
                    <th>{t("viewHeaders.state")}</th>
                </tr>
            </thead>
            <tbody key="body">
                {_map(views, (each) => {
                    return <SampleViewsRow key={each.ID} view={each} />;
                })}
            </tbody>
        </table>
    );
};

const SampleViewsModal = connect((state) => ({
    api: getApiState(state),
    visible: getSamplesState(state).modals.views,
    createViewFormVisible: getSamplesState(state).createSampleViewFormVisible,
    loaded: getSamplesState(state).loadedViews,
    failed: getSamplesState(state).failedViews,
    views: getSamplesState(state).views,
}))((props) => {
    const { sample, api, visible, createViewFormVisible, loaded, failed, views } = props;

    const dispatch = useDispatch();
    const { t } = useTranslation("samples");

    const fetchViews_ = () => dispatch(fetchViews(api, sample));

    useEffect(() => {
        if (sample) {
            fetchViews_();
        }
    }, [sample]);

    return (
        <div className="SampleViewsModal-container">
            <Modal
                view="default"
                theme="normal"
                visible={visible}
                zIndex={1}
                onClose={() => dispatch(closeSampleViews())}
            >
                {visible && (
                    <div className="SampleViews">
                        <LabModalHeader
                            title={`${t("viewsOf")} ${sample.name}`}
                            info={
                                <div className="SampleViews-header-help">
                                    <Link href={MATCHING_WIKI_URL} target="_blank">
                                        <img alt="?" src={helpPic} />
                                    </Link>
                                </div>
                            }
                            hasClose
                            onClose={() => dispatch(closeSampleViews())}
                        />

                        <Failable failed={failed} retry={fetchViews_}>
                            <Loading loaded={loaded}>
                                <SampleViewsTable views={views} />

                                {createViewFormVisible ? (
                                    <CreateViewSection sample={sample} />
                                ) : (
                                    <div className="SampleViewsModal-show-create-view-section-button">
                                        <Button
                                            view="action"
                                            theme="action"
                                            size="m"
                                            onClick={() => dispatch(setCreateSampleViewFormVisibility(true))}
                                        >
                                            {t("views.create")}
                                        </Button>
                                    </div>
                                )}
                            </Loading>
                        </Failable>
                    </div>
                )}
            </Modal>

            <SampleViewTableModal />
        </div>
    );
});

export default SampleViewsModal;
