import { createAction } from "redux-actions";
import _ from "lodash";
import store from "store";
import { reachGoal } from "metrika";
import { push } from "connected-react-router";
import { addErrorNotification, addSuccessNotification } from "../utils/notifications/notificationsState";

const MAX_EXTERNAL_SAMPLES_COUNT = 50;

export const requestSamples = createAction("request samples");
export const receiveSamples = createAction("receive samples");
export const failedSamples = createAction("failed samples");

export const openSampleStats = createAction("open sample stats");
export const closeSampleStats = createAction("close sample stats");
export const openSampleInfo = createAction("open sample info");
export const closeSampleInfo = createAction("close sample info");
export const openSampleViews = createAction("open sample views");
export const closeSampleViews = createAction("close sample views");

export const openSampleTableView = createAction("open sample table view");
export const closeSampleTableView = createAction("close sample table view");

export const openSiberiaSampleView = createAction("open siberia sample view");
export const closeSiberiaSampleView = createAction("close siberia sample view");

export const openSampleCreateModal = createAction("open sample import");
export const closeSampleCreateModal = createAction("close sample import");

export const setCreateSampleViewFormVisibility = createAction("show create sample view form");

export const requestSampleViews = createAction("request sample views");
export const receiveSampleViews = createAction("receive sample views");
export const failedSampleViews = createAction("failed sample views");

export const requestBaseSample = createAction("request base sample");
export const receiveBaseSample = createAction("receive base sample");
export const failedBaseSample = createAction("failed base sample");

export const updateBaseType = createAction("set base type");

export const changeSampleType = createAction("change sample type");

export const requestSampleStats = createAction("request sample stats");
export const receiveSampleStats = createAction("receive sample stats");
export const failedSampleStats = createAction("failed sample stats");
export const setSampleStatsNotFound = createAction("set sample stats not found");
export const setSampleStatsNotReady = createAction("set sample stats not ready");

export const selectSampleGroup = createAction("select sample group");

export const requestCreateView = createAction("request create view");
export const successCreateView = createAction("success create view");
export const failCreateView = createAction("failed create view");

export const requestSamplePreview = createAction("request sample page from yt");
export const receiveSamplePreview = createAction("receive sample page from yt");
export const failedSamplePreview = createAction("failed sample page from yt");

export const requestSiberiaSampleView = createAction("request sample from siberia");
export const receiveSiberiaSampleView = createAction("receive sample from siberia");
export const failedSiberiaSampleView = createAction("failed sample from siberia");

export const requestSiberiaSegments = createAction("request segments from siberia");
export const receiveSiberiaSegments = createAction("receive segments from siberia");
export const failedSiberiaSegments = createAction("failed segments from siberia");

export const selectSiberiaSegment = createAction("select siberia segment");
export const setSiberiaSegmentCreationTitle = createAction("set siberia segment creation title");
export const setSiberiaSegmentCreationRule = createAction("set siberia segment creation rule");

export const requestSiberiaSegmentRemoving = createAction("request siberia segment removing");
export const failSiberiaSegmentRemoving = createAction("fail siberia segment removing");
export const successSiberiaSegmentRemoving = createAction("success siberia segment removing");

export function deleteSample(state, sample) {
    return (dispatch) => {
        dispatch(reachGoal("deleteSample"));
        return state.api
            .then((api) => api.apis.lab.deleteSample)
            .then((method) => method({ id: sample.id }))
            .then((response) => {
                if (sample.siberiaUserSetId) {
                    dispatch(removeSiberiaUserSet(state.api, sample.siberiaUserSetId));
                }
                dispatch(reachGoal("deleteSampleSuccess"));
                dispatch(selectSample(undefined));
                dispatch(fetchSamples(state));
            })
            .catch((error) => {
                console.error("Failed to delete sample", sample.id);
                dispatch(reachGoal("deleteSampleFail"));
            });
    };
}

export function learnSample(state, sample) {
    return (dispatch) => {
        dispatch(reachGoal("learnSample"));
        return state.api
            .then((api) => api.apis.lab.runLearningTask)
            .then((method) => method({ id: sample.id }))
            .then((response) => {
                dispatch(addSuccessNotification("Learning task started"));
            })
            .catch((error) => {
                dispatch(addErrorNotification(error, "Failed to run learning task"));
            });
    };
}

export function selectSample(id) {
    return (dispatch) => {
        dispatch(push({ search: "?sample=" + id }));
    };
}

export function selectType(type) {
    return (dispatch) => {
        dispatch(selectSample(undefined));
        dispatch(changeSampleType(type));
    };
}

export function selectAndOpenSampleStats(id) {
    return (dispatch) => {
        dispatch(openSampleStats());
        dispatch(reachGoal("openSampleStats"));
        dispatch(selectSample(id));
    };
}

export function selectAndOpenSampleInfo(id) {
    return (dispatch) => {
        dispatch(openSampleInfo());
        dispatch(reachGoal("openSampleInfo"));
        dispatch(selectSample(id));
    };
}

export function selectAndOpenSampleViews(id) {
    return (dispatch) => {
        dispatch(openSampleViews());
        dispatch(reachGoal("openSampleViews"));
        dispatch(selectSample(id));
    };
}

export function selectAndOpenSiberiaSampleView(id) {
    return (dispatch) => {
        dispatch(openSiberiaSampleView());
        dispatch(reachGoal("openSiberiaView"));
        dispatch(selectSample(id));
    };
}

export function createSample(api, lang, setState, state) {
    return (dispatch) => {
        // hacky way, refactor later
        setState({ loaded: false, allFieldsDisabled: true, response: null }, () => {
            if (state.name && state.path) {
                api.then((api) => {
                    const params = {
                        name: state.name,
                        path: state.path,
                    };

                    if (state.idType) {
                        params["idType"] = state.idType;
                    }

                    if (state.idKey) {
                        params["idKey"] = state.idKey;
                    } else {
                        params["idKey"] = state.idType;
                    }

                    if (state.ttl && Number(state.ttl) > 0) {
                        params["ttl"] = Number(state.ttl) * 24 * 60 * 60;
                    }

                    if (state.groupingKey) {
                        params["groupingKey"] = state.groupingKey;
                    }

                    if (state.accessLevel) {
                        params["accessLevel"] = state.accessLevel;
                    }

                    return api.apis.lab.createSample(params);
                })
                    .then((response) => {
                        dispatch(reachGoal("createSampleSuccess"));
                        dispatch(closeSampleCreateModal());
                        dispatch(selectSample(response.obj.id));
                        dispatch(changeSampleType("my"));
                        dispatch(
                            fetchSamples({
                                api: api,
                                lang: lang,
                                selected: response.obj.id,
                            })
                        );
                    })
                    .catch((error) => {
                        dispatch(reachGoal("createSampleFail"));
                        setState({
                            error: error,
                            loaded: true,
                            allFieldsDisabled: false,
                            response: null,
                        });
                    });
            }
        });
    };
}

export function fetchSamples(state) {
    return (dispatch) => {
        dispatch(requestSamples());
        const { api } = state;
        let lastSeen = store.get("last_seen_external_samples", {});

        if (state.selected) {
            lastSeen[state.selected] = Date.now();
        }

        let fetches = [];
        fetches.push(
            api
                .then((api) => api.apis.lab.getSamples)
                .then((method) => method({ lang: state.lang }))
                .then((response) =>
                    _(response.obj)
                        .sortBy((sample) => sample["timestamps"]["modified"])
                        .map((each) => ({ type: "my", sample: each }))
                        .reverse()
                        .value()
                )
        );
        fetches.push(
            ..._(lastSeen)
                .toPairs()
                .orderBy((each) => each[1])
                .slice(-MAX_EXTERNAL_SAMPLES_COUNT)
                .reverse()
                .map((pair) =>
                    api
                        .then((api) => api.apis.lab.getSample({ id: pair[0] }))
                        .then((response) => response.obj)
                        .catch((error) => null)
                )
                .map((each) => each.then((value) => [{ type: "viewed", sample: value }]))
        );
        return Promise.all(fetches)
            .then((values) => {
                let r = { my: [], common: [], viewed: [] };
                for (let items of values) {
                    for (let item of items) {
                        if (item.sample) {
                            let type = item.type;
                            if (item.type === "my" && item.sample.accessLevel === "PUBLIC") {
                                type = "common";
                            }
                            r[type].push(item.sample);
                        }
                    }
                }
                return r;
            })
            .then((samples) => {
                // TODO fix this not the best code
                let myIds = new Set(samples.my.map((x) => x.id));
                let commonIds = new Set(samples.common.map((x) => x.id));
                samples.viewed = samples.viewed.filter((x) => !myIds.has(x.id) && !commonIds.has(x.id));
                let selectedViewed = samples.viewed.some((x) => x.id === state.selected);
                if (selectedViewed) {
                    dispatch(changeSampleType("viewed"));
                }
                let selectedCommon = samples.common.some((x) => x.id === state.selected);
                if (selectedCommon) {
                    dispatch(changeSampleType("common"));
                }
                let viewedIds = new Set(samples.viewed.map((x) => x.id));
                lastSeen = _.pickBy(lastSeen, (v, k) => viewedIds.has(k));
                store.set("last_seen_external_samples", lastSeen);

                return samples;
            })
            .then((data) => {
                dispatch(receiveSamples(data));
            })
            .catch((error) => dispatch(failedSamples(error)));
    };
}

export function fetchViews(api, sample) {
    return (dispatch) => {
        dispatch(requestSampleViews());
        return api
            .then((api) => api.apis.lab.getSampleViews)
            .then((method) => method({ id: sample.id }))
            .then((views) => dispatch(receiveSampleViews(views.obj)))
            .catch((error) => dispatch(failedSampleViews(error)));
    };
}

export function fetchBaseSample(api, baseSampleId) {
    return (dispatch) => {
        dispatch(requestBaseSample());
        return api
            .then((api) => api.apis.lab.getSample({ id: baseSampleId }))
            .then((response) => dispatch(receiveBaseSample(response.obj)))
            .catch((error) => dispatch(failedBaseSample(error)));
    };
}

export function changeBaseType() {
    return (dispatch) => {
        dispatch(updateBaseType());
    };
}

export const requestSubsamples = createAction("request subsamples");
export const receiveSubsamples = createAction("receive subsamples");
export const failSubsamples = createAction("fail subsamples");

export function fetchSubsamples(api, sampleId) {
    return (dispatch) => {
        dispatch(requestSubsamples());
        return api
            .then((api) => api.apis.lab.getSubsamples({ id: sampleId }))
            .then((response) => {
                dispatch(receiveSubsamples(response.obj));
            })
            .catch((error) => {
                dispatch(failSubsamples());
                dispatch(addErrorNotification(error.response), "Failed to fetch subsamples");
            });
    };
}

function createViewApiCall(api, sample, parameters) {
    if (parameters.type === "matching") {
        return api.then((api) => api.apis.lab.createView).then((method) => method(parameters));
    }
    if (parameters.type === "crypta_id_statistics") {
        return api
            .then((api) => api.apis.lab.createSpecialView)
            .then((method) => method({ mode: "CRYPTA_ID_STATISTICS", id: parameters.id }));
    }
    if (parameters.type === "lookalike") {
        return api
            .then((api) => api.apis.lab.createLookalikeView)
            .then((method) =>
                method({
                    id: parameters.id,
                    outputCount: Number(parameters.count),
                    useDates: parameters.useDates,
                })
            );
    }
    return Promise.reject("Wrong type");
}

export function createView(api, sample, parameters) {
    return (dispatch) => {
        dispatch(requestCreateView());

        createViewApiCall(api, sample, parameters)
            .then((response) => {
                dispatch(successCreateView());
                dispatch(fetchViews(api, sample));
            })
            .catch((error) => {
                let errorCode = null;
                if (error.obj && error.obj.code) {
                    errorCode = error.obj.code;
                }
                dispatch(failCreateView(errorCode));
                dispatch(addErrorNotification(error, "Failed to create view of sampleId=" + sample.id));
            });
    };
}

export function openSampleTablePreview(api, sampleId, viewId) {
    return (dispatch) => {
        dispatch(openSampleTableView(sampleId, viewId));
        dispatch(fetchSamplePreview(api, sampleId, viewId));
    };
}

export function fetchSamplePreview(api, sampleId, viewId, pageNumber = 1, pageSize = 20) {
    return (dispatch) => {
        dispatch(requestSamplePreview());
        return api
            .then((api) => api.apis.lab.getSampleViewFromYtByPages)
            .then((method) =>
                method({
                    sample_id: sampleId,
                    view_id: viewId,
                    page: pageNumber,
                    page_size: pageSize,
                })
            )
            .then((sample) => {
                dispatch(receiveSamplePreview(sample.obj));
            })
            .catch((error) => dispatch(failedSamplePreview()));
    };
}

export function fetchSiberiaSample(api, siberiaUserSetId, lastUserId) {
    let params = {
        user_set_id: siberiaUserSetId,
    };

    if (lastUserId) {
        params.last_user_id = lastUserId;
    }

    return (dispatch) => {
        dispatch(requestSiberiaSampleView(siberiaUserSetId));
        return api
            .then((api) => api.apis.lab.getSampleFromSiberia(params))
            .then((sample) => {
                dispatch(receiveSiberiaSampleView(sample.obj));
            })
            .catch((error) => dispatch(failedSiberiaSampleView(error)));
    };
}

export function fetchSiberiaSegments(api, siberiaUserSetId, lastSegmentId) {
    let params = {
        user_set_id: siberiaUserSetId,
        limit: 1000,
    };

    if (lastSegmentId) {
        params.last_segment_id = lastSegmentId;
    }

    return (dispatch) => {
        dispatch(requestSiberiaSegments(siberiaUserSetId));
        return api
            .then((api) => {
                return api.apis.lab.getSegmentsFromSiberia(params);
            })
            .then((response) => {
                dispatch(receiveSiberiaSegments(response.obj));
            })
            .catch((error) => {
                if (error.errObj && error.errObj.statusCode === 404) {
                    dispatch(receiveSiberiaSegments([]));
                } else {
                    dispatch(failedSiberiaSegments(error));
                }
            });
    };
}

export function createSiberiaSegment(api, params, onError, onSuccess) {
    return (dispatch) => {
        api.then((api) => {
            return api.apis.lab.createSiberiaSegment(params);
        })
            .then((response) => {
                onSuccess(response);
                let segment = response.obj;
                dispatch(selectSiberiaSegment(segment));
                dispatch(getSiberiaSegmentUsers(api, params.user_set_id, segment.Id));
                dispatch(fetchSiberiaSegments(api, params.user_set_id));
            })
            .catch((error) => {
                onError(error);
            });
    };
}

export function removeSiberiaSegment(api, userSetId, segmentId, onError) {
    return (dispatch) => {
        api.then((api) => {
            requestSiberiaSegmentRemoving();
            return api.apis.lab.removeSiberiaSegment({
                user_set_id: userSetId,
                segment_id: segmentId,
            });
        })
            .then((response) => {
                successSiberiaSegmentRemoving();
                dispatch(fetchSiberiaSegments(api, userSetId));
            })
            .catch((error) => {
                failSiberiaSegmentRemoving();
                onError(error);
            });
    };
}

export function getSiberiaSegmentUsers(api, userSetId, segmentId, lastUserId) {
    let params = {
        user_set_id: userSetId,
        segment_id: segmentId,
    };

    if (lastUserId) {
        params.last_user_id = lastUserId;
    }

    return (dispatch) => {
        dispatch(requestSiberiaSampleView(userSetId));
        api.then((api) => {
            return api.apis.lab.getSiberiaSegmentUsers(params);
        })
            .then((response) => {
                dispatch(receiveSiberiaSampleView(response.obj));
            })
            .catch((error) => {
                dispatch(failedSiberiaSampleView(error));
            });
    };
}

export function removeSiberiaUserSet(api, userSetId) {
    let params = {
        user_set_id: userSetId,
    };

    return (dispatch) => {
        api.then((api) => {
            return api.apis.lab.removeSiberiaUserSet(params);
        })
            .then((response) => {})
            .catch((error) => {});
    };
}
