import React, { useEffect } from "react";
import { isEmpty, isUndefined } from "lodash";
import SimpleDate from "../utils/SimpleDate";
import FontAwesome from "react-fontawesome";
import { useTranslation } from "react-i18next";
import Loading from "@lib/components/loading";
import SamplesMenu from "./SamplesMenu";
import SampleInfoModal from "./SampleInfoModal";
import SampleViewsModal from "./SampleViewsModal";
import DatedSampleViewsModal from "../predictive_analytics/components/DatedSampleViewsModal";
import DatedSampleStatsModal from "../predictive_analytics/components/DatedSampleStatsModal";
import SampleStatus from "../predictive_analytics/components/SampleStatus";
import Failable from "../utils/Failable";
import Id from "../utils/Id";
import User from "../utils/User";
import { connect } from "react-redux";
import { getAccessState, getApiState, getSamplesState, getSelectedSampleIdState } from "../selectors";
import {
    deleteSample,
    fetchSamples,
    learnSample,
    selectAndOpenSampleInfo,
    selectAndOpenSampleStats,
    selectAndOpenSampleViews,
    selectAndOpenSiberiaSampleView,
    selectSample,
} from "./actions";
import SiberiaSampleViewModal from "src/samples/SiberiaSampleViewModal";
import LabLink from "../utils/LabLink";
import SampleStatsModal_ from "./stats";

import "./Samples.scss";

const SamplesTableRow = connect((state) => ({
    api: getApiState(state),
    selected: getSelectedSampleIdState(state),
    access: getAccessState(state),
}))((props) => {
    const { api, selected, access, sample, dispatch, dated } = props;

    const { t, i18n } = useTranslation("samples");
    const lang = i18n.language;

    const showSampleStats = () => {
        dispatch(selectAndOpenSampleStats(sample.id));
    };

    const isSelected = sample.id === selected;

    return (
        <tr
            className={`SamplesTableRow ${isSelected ? "selected" : ""}`}
            onClick={() => dispatch(selectSample(sample.id))}
        >
            <td key="id" className="SampleId">
                <Id copyable={true} value={sample.id} />
            </td>
            <td key="name" className="SampleName" title={sample.name.length > 30 ? sample.name : ""}>
                {sample.name}
            </td>
            <td key="type" className="SampleType">
                <Id value={sample.idName} />
            </td>
            <td key="author" className="SampleAuthor">
                <User login={sample.author} />
            </td>
            <td key="modified" className="SampleModified">
                <SimpleDate lang={lang} value={sample.timestamps.modified} />
            </td>
            <td key="accessLevel" className="SampleAccessLevel">
                <div className="SampleAccessLevel-content">
                    {t(`access.${sample.accessLevel}`)}
                    {["SHARED", "PUBLIC"].includes(sample.accessLevel) && <LabLink search={"sample=" + sample.id} />}
                </div>
            </td>
            {dated && (
                <td key="status" className="SampleStatus">
                    <SampleStatus state={sample.state} onClick={() => dispatch(selectAndOpenSampleViews(sample.id))}/>
                </td>
            )}
            <td key="buttons">
                {access.Siberia && (
                    <span
                        className="SampleButton"
                        onClick={() => dispatch(selectAndOpenSiberiaSampleView(sample.id))}
                        title={t("headers.siberiaPreview")}
                    >
                        <FontAwesome name="search" />
                    </span>
                )}
                {
                    // temporarily disable learning button - CRYPTA-14309
                    false && access.Learning && (
                        <FontAwesome
                            key="learning"
                            className="SampleButton"
                            name="university"
                            title={t("headers.learning")}
                            onClick={() => dispatch(learnSample({ api: api }, sample))}
                        />
                    )
                }
                <FontAwesome
                    key="info"
                    className="SampleButton"
                    name="info-circle"
                    title={t("headers.info")}
                    onClick={() => dispatch(selectAndOpenSampleInfo(sample.id))}
                />
                <FontAwesome
                    key="stats"
                    className="SampleButton"
                    name="bar-chart"
                    title={t("headers.stats")}
                    onClick={showSampleStats}
                />
                <FontAwesome
                    key="matching"
                    className="SampleButton"
                    name="random"
                    title={t("headers.matching")}
                    onClick={() => dispatch(selectAndOpenSampleViews(sample.id))}
                />
                <FontAwesome
                    key="remove"
                    className="SampleButton"
                    name="remove"
                    title={t("headers.deletion")}
                    onClick={() => {
                        const message = t("remove") + " " + sample.id + "?";
                        if (window.confirm(message)) {
                            dispatch(
                                deleteSample(
                                    {
                                        api: api,
                                        lang: lang,
                                        selected: selected,
                                    },
                                    sample
                                )
                            );
                        }
                    }}
                />
            </td>
        </tr>
    );
});

const SamplesTableHeader = (props) => {
    const { t } = useTranslation("samples");

    return (
        <tr className="SamplesTableHeader">
            <th key="id">{t("headers.id")}</th>
            <th key="name">{t("headers.name")}</th>
            <th key="type">{t("headers.type")}</th>
            <th key="author">{t("headers.author")}</th>
            <th key="modified">{t("headers.modified")}</th>
            <th key="accessLevel">{t("headers.access")}</th>
            {props.dated && <th key="state" className="SamplesTableHeader-center">{t("headers.status")}</th>}
            <th key="buttons"> </th>
        </tr>
    );
};

const Navigation = connect((state) => ({
    samples: getSamplesState(state).samples,
    type: getSamplesState(state).type,
}))((props) => {
    const { samples, type, dated } = props;

    return (
        <div className="Navigation">
            <SamplesMenu dated={dated} />
            <table className="SamplesTable">
                <thead>
                    <SamplesTableHeader key="header" dated={dated} />
                </thead>
                <tbody>
                    {
                        // Filter condition is true in 2 cases:
                        // 1. if samples dateKey is "" and simple samples tab is currently viewed
                        // 2. if samples dateKey is not empty  and dated samples tab is currently viewed
                        samples[type].filter((each) => {return isEmpty(each.dateKey) === isUndefined(dated)}).map((each) => {
                            return <SamplesTableRow key={each.id} sample={each} dated={dated} />;
                        })
                    }
                </tbody>
            </table>
        </div>
    );
});

function selectedSample(samples, type, selected) {
    return samples[type].filter((each) => each.id === selected)[0];
}

const Samples = connect((state) => ({
    api: getApiState(state),
    failed: getSamplesState(state).failed,
    loaded: getSamplesState(state).loaded,
    samples: getSamplesState(state).samples,
    message: getSamplesState(state).message,
    modals: getSamplesState(state).modals,
    selected: getSelectedSampleIdState(state),
    type: getSamplesState(state).type,
    location: state.router.location,
    siberiaSampleViewVisible: getSamplesState(state).modals.siberiaView,
}))((props) => {
    const { api, samples, type, selected, failed, loaded, message, siberiaSampleViewVisible, dispatch, dated } = props;

    const { i18n } = useTranslation();
    const lang = i18n.language;

    const fetch = () => {
        dispatch(
            fetchSamples({
                api: api,
                lang: lang,
                selected: selected,
            })
        );
    };

    useEffect(() => {
        fetch();
    }, []);

    let sample = selectedSample(samples, type, selected);

    return (
        <Failable failed={failed} errorMessage={message} retry={fetch}>
            <Loading loaded={loaded}>
                <div className="Samples">
                    <Navigation dated={dated} />
                    <SampleInfoModal sample={sample} />
                    {dated ? (
                        <div>
                            <DatedSampleViewsModal sample={sample} />
                            <DatedSampleStatsModal />
                        </div>
                        ) : (
                            <div>
                                <SampleViewsModal sample={sample} />
                                <SampleStatsModal_ />
                            </div>
                        )
                    }
                    {!dated && siberiaSampleViewVisible && <SiberiaSampleViewModal sample={sample} lang={lang} />}
                </div>
            </Loading>
        </Failable>
    );
});

export default Samples;
