import React from "react";
import { connect } from "react-redux";
import { isEmpty, isNull } from "lodash";
import { useTranslation } from "react-i18next";
import { Modal } from "@lib/components/lego";
import SampleStats from "@lib/stats/SampleStats";

import {
    fetchRelativeSegmentStats,
    hideRuleEstimateStatsModal,
    hideRuleRevisionEstimateStatsModal,
    hideSegmentStatsModal
} from "./actions";
import {
    getApiState,
    getVisibleSegmentStatsModalState,
    getActiveSegmentAttributesState,
    getActiveSegmentExportState,
    getActiveConditionState,
    getRuleEstimateStatsState,
    getRuleRevisionEstimateStatsState,
    getRuleState,
    getVisibleRuleEstimateStatsModalState,
    getVisibleRuleRevisionEstimateStatsModalState,
} from "../selectors";
import Failable from "../utils/Failable";
import { getSegmentStatsState } from "./selectors";
import LabModalHeader from "../utils/LabModalHeader";
import BaseStatsParams from "../stats/BaseStatsParams";

import "./style.scss";


function makeModal(stats, segmentExport, visible, title, info, exportFileBasename, loading, failed, t, hideStatsModal, api) {
    let actualFailed = failed;
    let actualStatusMessage = t("stats.error");

    if (!failed && stats && stats.Info && !stats.Info.Ready) {
        actualFailed = true;
        actualStatusMessage = t("stats.notReady");
    }

    return (
        visible && (
            <Modal theme="normal" onClose={hideStatsModal} visible={visible}>
                <div className="SegmentStats">
                    <LabModalHeader title={title} onClose={hideStatsModal} info={info} hasClose />
                    {!isNull(segmentExport) && (
                        <BaseStatsParams
                            fetchStats={() => fetchRelativeSegmentStats(api, segmentExport.id)}
                            fetchStatsBySample={
                                (baseId, baseGroup, baseSetId) => fetchRelativeSegmentStats(api, segmentExport.id, null, baseId, baseGroup)
                            }
                            fetchStatsByExport={
                                (baseID) => fetchRelativeSegmentStats(api, segmentExport.id, baseID, null, null)
                            }
                        />
                    )}
                    <div key="stats" className="SegmentStats-content">
                        <Failable failed={actualFailed} errorMessage={actualStatusMessage}>
                            {!isEmpty(stats) && (
                                <SampleStats
                                    loaded={!loading}
                                    stats={{
                                        value: stats.Stats,
                                        ready: stats.Info.Ready,
                                    }}
                                    exportFileBasename={exportFileBasename}
                                    withLoading={false}
                                />
                            )}
                        </Failable>
                    </div>
                </div>
            </Modal>
        )
    );
}

const StatsModal = connect((state) => ({
    api: getApiState(state),
    segment: getActiveSegmentAttributesState(state),
    segmentExport: getActiveSegmentExportState(state),
    visible: getVisibleSegmentStatsModalState(state),
    stats: getSegmentStatsState(state).value,
    loading: getSegmentStatsState(state).loading,
    failed: getSegmentStatsState(state).error,
}))((props) => {
    const { api, stats, visible, segment, segmentExport, loading, failed, dispatch } = props;

    const { t } = useTranslation("segments");

    const hideStatsModal = () => dispatch(hideSegmentStatsModal());

    let title = segment ? segment.name : "";
    let info = segment ? segment.id : "";
    let exportFileBasename = segment && segmentExport ? segment.id + "__" + segmentExport.id : "";

    return makeModal(stats, segmentExport, visible, title, info, exportFileBasename, loading, failed, t, hideStatsModal, api);
});

const RuleConditionEstimateStatsModal = connect((state) => ({
    api: getApiState(state),
    activeCondition: getActiveConditionState(state),
    visible: getVisibleRuleRevisionEstimateStatsModalState(state),
    stats: getRuleRevisionEstimateStatsState(state).value,
    loading: getRuleRevisionEstimateStatsState(state).loading,
    failed: getRuleRevisionEstimateStatsState(state).error,
}))((props) => {
    const { api, stats, visible, loading, failed, activeCondition, dispatch } = props;

    const { t } = useTranslation("segments");

    const hideStatsModal = () => dispatch(hideRuleRevisionEstimateStatsModal());

    const conditionSource = activeCondition && activeCondition.source ? activeCondition.source : "";

    let title = !isEmpty(conditionSource) ? t("condition:sourcesTitles." + conditionSource) : "";
    let info = activeCondition ? activeCondition.revision : "";
    let exportFileBasename = activeCondition
        ? activeCondition.revision + "__" + conditionSource + "__" + activeCondition.state
        : "";

    return makeModal(stats, null, visible, title, info, exportFileBasename, loading, failed, t, hideStatsModal, api);
});

const RuleEstimateStatsModal = connect((state) => ({
    api: getApiState(state),
    rule: getRuleState(state),
    visible: getVisibleRuleEstimateStatsModalState(state),
    stats: getRuleEstimateStatsState(state).value,
    loading: getRuleEstimateStatsState(state).loading,
    failed: getRuleEstimateStatsState(state).error,
}))((props) => {
    const { api, stats, visible, loading, failed, rule, dispatch } = props;

    const { t } = useTranslation("segments");

    const hideStatsModal = () => dispatch(hideRuleEstimateStatsModal());

    let title = rule ? rule.name : "";
    let info = rule ? rule.id : "";
    let exportFileBasename = rule ? rule.id + "__" + rule.name : "";

    return makeModal(stats, null, visible, title, info, exportFileBasename, loading, failed, t, hideStatsModal, api);
});

export default StatsModal;
export { RuleConditionEstimateStatsModal, RuleEstimateStatsModal };
