import { createAction } from "redux-actions";
import { reachGoal } from "metrika";
import { isEmpty } from "lodash";
import { push } from "connected-react-router";
import qs from "query-string";
import { addErrorNotification, addSuccessNotification } from "../utils/notifications/notificationsState";
import { getQueryState } from "./selectors";

export const receiveGroups = createAction("receive segments groups");
export const receiveSegments = createAction("receive segments");
export const receiveSegmentsDone = createAction("receive segments done");
export const receiveRule = createAction("receive rule");
export const failedSegments = createAction("failed segments");

export const requestNotLocalizedSegment = createAction("request not localized segment");
export const receiveNotLocalizedSegment = createAction("store not localized segment");
export const failNotLocalizedSegment = createAction("fail not localized segment");

export const showSegmentStatsModal = createAction("show SegmentStatsModal");
export const hideSegmentStatsModal = createAction("hide SegmentStatsModal");
export const showRuleRevisionEstimateStatsModal = createAction("show RuleRevisionEstimateStatsModal");
export const hideRuleRevisionEstimateStatsModal = createAction("hide RuleRevisionEstimateStatsModal");
export const showRuleEstimateStatsModal = createAction("show RuleEstimateStatsModal");
export const hideRuleEstimateStatsModal = createAction("hide RuleEstimateStatsModal");
export const showRuleModal = createAction("show RuleModal");
export const hideRuleModal = createAction("hide RuleModal");
export const refreshRule = createAction("refresh rule");
export const handleSearch = createAction("handle search");
export const resetSearch = createAction("reset search");
export const updateSegmentToMove = createAction("update segment to move");
export const handleEditButtonClick = createAction("handle edit button click");
export const startProcessingData = createAction("start processing data");
export const finishProcessingData = createAction("finish processing data");
export const failProcessingData = createAction("fail processing data");

export const setExpressionsModalVisibility = createAction("set expressions modal visibility");

export const setLalModalVisibility = createAction("set lal modal visibility");

export const setCreateSegmentModalVisibility = createAction("set create segment modal visibility");

export const requestSegmentExportTags = createAction("segment export tags requested");
export const receiveSegmentExportTags = createAction("segment export tags received");
export const failSegmentExportTags = createAction("segment export tags failed");
export const openExportTagsModal = createAction("open export tags modal");
export const closeExportTagsModal = createAction("close export tags modal");
export const setActiveSegmentExportId = createAction("set active segment export id");
export const setActiveSegmentExportTags = createAction("set active segment export tags");

export const receiveExportsToSegments = createAction("receive exports to segments");

export const requestRequestConditionApproval = createAction("request request condition approval");
export const receiveRequestConditionApproval = createAction("receive request condition approval");
export const failRequestConditionApproval = createAction("fail request condition approval");
export const resetRequestConditionApproval = createAction("reset request condition approval");
export const setRequestApprovalVisibility = createAction("set request approval visibility");

export const setTankerModalVisibility = createAction("set tanker modal visibility");

export const requestSegmentsAttributes = createAction("request segments attributes");
export const receiveSegmentsAttributes = createAction("receive segments attributes");
export const failSegmentsAttributes = createAction("fail segments attributes");
export function fetchSegmentsAttributes(api, lang) {
    return (dispatch, getState) => {
        dispatch(requestSegmentsAttributes());

        return api
            .then((api) => {
                const query = getQueryState(getState());

                if (!isEmpty(query)) {
                    return api.apis.lab.matchSegmentsSearchItems({ searchItems: query.split(" ") });
                } else {
                    return api.apis.lab.getSegmentsAttributes({ lang: lang });
                }
            })
            .then((response) => {
                return dispatch(receiveSegmentsAttributes(response.obj));
            })
            .catch((error) => {
                dispatch(failSegmentsAttributes());
                dispatch(addErrorNotification(error, "Failed fetching segments"));
            });
    };
}

export const requestSegmentsData = createAction("request segments data with attributes");
export const receiveSegmentsData = createAction("receive segments data with attributes");
export const failSegmentsData = createAction("fail segments data with attributes");
export function fetchSegmentsAndGroups(api, lang) {
    return (dispatch) => {
        dispatch(requestSegmentsData());
        return api
            .then((api) => {
                return api.apis.lab.getAllSegmentGroups({ lang: lang });
            })
            .then((response) => {
                dispatch(receiveGroups(response.obj));
            })
            .then((groupsDone) => {
                return dispatch(fetchSegmentsAttributes(api, lang));
            })
            .then((segmentsDone) => dispatch(receiveSegmentsDone()))
            .catch((error) => {
                dispatch(failSegmentsData());
            });
    };
}

// SegmentAttributes in https://a.yandex-team.ru/arc_vcs/crypta/lab/proto/segment.proto
const segmentAttributesFields = ["id", "name", "parentId", "priority", "type"];
export const requestSegmentInfo = createAction("request segment info");
export const receiveSegmentInfo = createAction("receive segment info");
export const failSegmentInfo = createAction("fail segment info");

export function fetchSegment(api, lang, segmentId) {
    return (dispatch) => {
        dispatch(requestSegmentInfo());

        return api
            .then((api) => api.apis.lab.getSegment)
            .then((method) => method({ lang: lang, id: segmentId }))
            .then((response) => {
                let segment = response.obj;
                let segmentAttributes = {};
                segmentAttributesFields.forEach((field) => {
                    segmentAttributes[field] = segment[field];
                });

                dispatch(receiveSegmentInfo(segment));
            })
            .catch((error) => {
                dispatch(failSegmentInfo());
                dispatch(addErrorNotification(error.response, "Failed to fetch " + segmentId));
            });
    };
}

export function fetchNotLocalizedSegment(api, segmentId) {
    return (dispatch) => {
        dispatch(requestNotLocalizedSegment());

        return api
            .then((api) => api.apis.lab.getNotLocalizedSegment)
            .then((method) => method({ id: segmentId }))
            .then((response) => dispatch(receiveNotLocalizedSegment(response.obj)))
            .catch((error) => {
                dispatch(failNotLocalizedSegment());
                dispatch(addErrorNotification(error, "Failed to fetch segment " + segmentId));
            });
    };
}

export const requestUpdateNameAndDescription = createAction("request update name and description");
export const receiveUpdateNameAndDescription = createAction("receive update name and description");
export const failUpdateNameAndDescription = createAction("fail updtate name and description");

export function updateNameAndDescription(api, lang, segmentId, name, description) {
    return (dispatch) => {
        dispatch(requestUpdateNameAndDescription());

        return api
            .then((api) => api.apis.lab.updateNameAndDescription)
            .then((method) => method({ id: segmentId, name: name, description: description }))
            .then((response) => {
                dispatch(receiveUpdateNameAndDescription(response.obj));
                dispatch(fetchSegment(api, lang, segmentId));
                dispatch(addSuccessNotification("Properties updated"));
            })
            .then((done) => {
                dispatch(fetchSegmentsAttributes(api, lang));
            })
            .catch((error) => {
                dispatch(failUpdateNameAndDescription());
                dispatch(addErrorNotification(error, "Failed to update properties"));
            });
    };
}

export const requestMatchSegmentConditions = createAction("request match segment conditions");
export const failMatchSegmentCondition = createAction("fail match segment conditions");
export const receiveMatchSegmentConditions = createAction("receive segments list");
export const resetMatchSegmentConditions = createAction("reset segments list");
export function matchSegmentConditions(api, conditions) {
    return (dispatch) => {
        dispatch(requestMatchSegmentConditions());

        return api
            .then((api) => api.apis.lab.matchSegmentsWithConditions)
            .then((method) => method({ conditions: JSON.stringify(conditions) }))
            .then((response) => {
                dispatch(receiveMatchSegmentConditions({
                    matched: response.obj,
                    showDeleted: conditions.showDeleted ? conditions.showDeleted : false,
                }));
            })
            .catch((error) => {
                dispatch(failMatchSegmentCondition());
                addErrorNotification(error, "Error finding segments");
            });
    };
}

export const requestMatchSearchItems = createAction("request match search items");
export const receiveMatchSearchItems = createAction("receive match search items");
export const failMatchSearchItems = createAction("fail match search items");

export function matchSearchItems(api, searchItems) {
    return (dispatch) => {
        dispatch(requestMatchSearchItems());

        return api
            .then((api) => api.apis.lab.matchSegmentsSearchItems)
            .then((method) => method({ searchItems: searchItems }))
            .then((response) => {
                dispatch(receiveMatchSearchItems(response.obj));
            })
            .then((done) => {
                dispatch(handleSearch(searchItems));
            })
            .catch((error) => {
                dispatch(failMatchSearchItems());
                addErrorNotification(error, "Error matching search items");
            });
    };
}

export const requestUpdateName = createAction("request update name");
export const receiveUpdateName = createAction("receive update name");
export const failUpdateName = createAction("fail update name");

export function updateName(api, lang, segmentId, name) {
    return (dispatch) => {
        dispatch(requestUpdateName());

        return api
            .then((api) => api.apis.lab.updateName)
            .then((method) => method({ id: segmentId, value: name }))
            .then((response) => {
                dispatch(receiveUpdateName(response.obj));
                dispatch(fetchSegment(api, lang, segmentId));
                dispatch(addSuccessNotification("Name updated"));
            })
            .catch((error) => {
                dispatch(failUpdateName());
                dispatch(addErrorNotification(error, "Failed to update name"));
            });
    };
}

export function addStakeholder(api, lang, segmentId, stakeholder) {
    return (dispatch) => {
        return api
            .then((api) => {
                return api.apis.lab.addStakeholder({
                    id: segmentId,
                    stakeholder: stakeholder,
                });
            })
            .then((response) => {
                dispatch(addSuccessNotification("Stakeholder added: " + stakeholder));
                dispatch(fetchSegment(api, lang, segmentId));
                dispatch(reachGoal("addStakeholderSuccess"));
            })
            .catch((error) => {
                dispatch(addErrorNotification(error, "Failed to add stakholder"));
                dispatch(reachGoal("addStakeholderFailed"));
            });
    };
}

export function removeStakeholder(api, lang, segmentId, stakeholder) {
    return (dispatch) => {
        return api
            .then((api) => {
                return api.apis.lab.deleteStakeholder({
                    id: segmentId,
                    stakeholder: stakeholder,
                });
            })
            .then((response) => {
                dispatch(addSuccessNotification("Stakeholder removed: " + stakeholder));
                dispatch(fetchSegment(api, lang, segmentId));
                dispatch(reachGoal("removeStakeholderSuccess"));
            })
            .catch((error) => {
                dispatch(addErrorNotification(error, "Failed to remove stakeholder"));
                dispatch(reachGoal("removeStakeholderFailed"));
            });
    };
}

export function addResponsible(api, lang, segmentId, responsible) {
    return (dispatch) => {
        return api
            .then((api) => {
                return api.apis.lab.addResponsible({
                    id: segmentId,
                    responsible: responsible,
                });
            })
            .then((response) => {
                dispatch(addSuccessNotification("Responsible added: " + responsible));
                dispatch(fetchSegment(api, lang, segmentId));
                dispatch(reachGoal("addResponsibleSuccess"));
            })
            .catch((error) => {
                dispatch(addErrorNotification(error, "Failed to add responsible"));
                dispatch(reachGoal("addResponsibleFailed"));
            });
    };
}

export function removeResponsible(api, lang, segmentId, responsible) {
    return (dispatch) => {
        return api
            .then((api) => {
                return api.apis.lab.deleteResponsible({
                    id: segmentId,
                    responsible: responsible,
                });
            })
            .then((response) => {
                dispatch(addSuccessNotification("Responsible removed: " + responsible));
                dispatch(fetchSegment(api, lang, segmentId));
                dispatch(reachGoal("removeResponsibleSuccess"));
            })
            .catch((error) => {
                dispatch(addErrorNotification(error, "Failed to remove responsible"));
                dispatch(reachGoal("removeResponsibleFailed"));
            });
    };
}

export function handleSegmentMove(api, lang, segmentToMove, newSegmentParent) {
    return (dispatch) => {
        return api
            .then((api) => {
                if (segmentToMove.id.startsWith("group-")) {
                    return api.apis.lab.setGroupParentId({
                        groupId: segmentToMove.id,
                        newParentGroupId: newSegmentParent.id,
                    });
                } else {
                    return api.apis.lab.setSegmentGroup({
                        segmentId: segmentToMove.id,
                        newGroupId: newSegmentParent.id,
                    });
                }
            })
            .then((response) => {
                dispatch(fetchSegmentsAttributes(api, lang));
                dispatch(updateSegmentToMove(undefined));
            })
            .catch((error) => {
                dispatch(addErrorNotification(error, "Failed to move segment"));
            });
    };
}

const updateActiveIds = (parent, segment) => {
    return (dispatch) => {
        let group = isEmpty(parent) ? "root-users" : parent;
        dispatch(
            push({
                search:
                    "?" +
                    qs.stringify({
                        group: group,
                        segment: segment,
                    }),
            })
        );
    };
};

export function createSegment(
    api,
    lang,
    name,
    nameEn,
    description,
    descriptionEn,
    tickets,
    segmentType,
    scope,
    parentId,
    responsibles,
    stakeholders
) {
    return (dispatch) => {
        return api
            .then((api) => api.apis.lab.createSegment)
            .then((method) => {
                return method({
                    nameRu: name,
                    nameEn: nameEn,
                    descriptionRu: description,
                    descriptionEn: descriptionEn,
                    tickets: tickets,
                    type: segmentType,
                    responsibles: responsibles,
                    stakeholders: stakeholders,
                    scope: scope,
                    parentId: parentId,
                });
            })
            .then((response) => {
                dispatch(reachGoal("createSegmentSuccess"));
                dispatch(fetchSegmentsAttributes(api, lang));
                return response;
            })
            .then((response) => {
                dispatch(updateActiveIds(parentId, response.obj.id));
            })
            .catch((error) => {
                dispatch(reachGoal("createSegmentFailed"));
                dispatch(addErrorNotification(error, "Failed to create segment"));
            });
    };
}

export function createUserSegment(api, lang, name, nameEn, description, descriptionEn, tickets) {
    return (dispatch) => {
        return api
            .then((api) => api.apis.lab.createUserSegment)
            .then((method) => {
                return method({
                    nameRu: name,
                    nameEn: nameEn,
                    descriptionRu: description,
                    descriptionEn: descriptionEn,
                    tickets: tickets,
                });
            })
            .then((response) => {
                dispatch(reachGoal("createUserSegmentSuccess"));
                dispatch(fetchSegmentsAttributes(api, lang));

                dispatch(updateActiveIds("root-users", response.obj.id));
            })
            .catch((error) => {
                dispatch(reachGoal("createUserSegmentFailed"));
                dispatch(addErrorNotification(error, "Failed to create user segment"));
            });
    };
}

export function createSegmentExportRule(api, lang, segmentId, exportId, name, days, minDays) {
    return (dispatch) => {
        return api
            .then((api) => api.apis.lab.createExportRule)
            .then((method) =>
                method({
                    id: exportId,
                    name: name,
                    days: days,
                    min_days: minDays,
                })
            )
            .then((response) => {
                dispatch(reachGoal("createSegmentExportRuleSuccess"));
                dispatch(fetchSegment(api, lang, segmentId));
                dispatch(showRuleModal(response.obj.id));
            })
            .catch((error) => {
                dispatch(reachGoal("createSegmentExportRuleFailed"));
                dispatch(addErrorNotification(error, "Failed to create rule"));
            });
    };
}

export function deleteSegmentExportRule(api, lang, segmentId, exportId) {
    return (dispatch) => {
        return api
            .then((api) => api.apis.lab.deleteExportRule)
            .then((method) => method({ id: exportId }))
            .then((response) => {
                dispatch(reachGoal("deleteSegmentExportRuleSuccess"));
                dispatch(fetchSegment(api, lang, segmentId));
                dispatch(hideRuleModal());
            })
            .catch((error) => {
                dispatch(reachGoal("deleteSegmentExportRuleFailed"));
                dispatch(addErrorNotification(error, "Failed to delete rule"));
            });
    };
}

export function updateRule(api, lang, ruleId, name, days, minDays) {
    return (dispatch) => {
        return api
            .then((api) => api.apis.lab.updateRule)
            .then((method) =>
                method({
                    id: ruleId,
                    name: name,
                    days: days,
                    min_days: minDays,
                })
            )
            .then((response) => {
                dispatch(refreshRule(response.obj));
                dispatch(reachGoal("updateRuleSuccess"));
            })
            .catch((error) => {
                dispatch(reachGoal("updateRuleFail"));
                dispatch(error, "Failed to update rule");
            });
    };
}

export function unlinkSegmentExportRule(api, lang, segmentId, exportId) {
    return (dispatch) => {
        return api
            .then((api) => api.apis.lab.deleteExportRuleId)
            .then((method) => method({ id: exportId }))
            .then((response) => {
                dispatch(reachGoal("unlinkExportRuleSuccess"));
                dispatch(fetchSegment(api, lang, segmentId));
                dispatch(hideRuleModal());
            })
            .catch((error) => {
                dispatch(reachGoal("unlinkExportRuleFail"));
                dispatch(addErrorNotification(error, "Failed to unlink rule"));
            });
    };
}

export function fetchRule(api, lang, id) {
    return (dispatch) => {
        return api
            .then((api) => api.apis.lab.getRule)
            .then((method) => method({ id: id }))
            .then((response) => {
                dispatch(receiveRule(response.obj));
            })
            .catch((error) => {
                dispatch(addErrorNotification(error, "Failed to fetch rule"));
            });
    };
}

export const setActiveCondition = createAction("set active condition");
export const setConditionErrorVisible = createAction("set condition error visible");

export function putRuleCondition(api, lang, ruleId, source, values, setVisibleError) {
    return (dispatch) => {
        return api
            .then((api) => api.apis.lab.putRuleCondition)
            .then((method) => method({ id: ruleId, source: source, values: values }))
            .then((response) => {
                dispatch(reachGoal("putRuleConditionSuccess"));
                dispatch(setActiveCondition(response.obj));
                if (response.obj.hasErrors) {
                    setVisibleError(true);
                }
                return dispatch(fetchRule(api, lang, ruleId));
            })
            .catch((error) => {
                dispatch(reachGoal("putRuleConditionFail"));
                dispatch(addErrorNotification(error, "Failed to add condition"));
            });
    };
}

export function requestConditionApproval(api, ruleId, author, linkedIssue, url) {
    return (dispatch) => {
        dispatch(requestRequestConditionApproval());

        return api
            .then((api) => {
                return api.apis.lab.requestRuleApproveViaTicket({
                    ruleId: ruleId,
                    author: author,
                    linkedIssue: linkedIssue,
                    url: url,
                });
            })
            .then((response) => {
                dispatch(receiveRequestConditionApproval(response.obj));
            })
            .catch((error) => {
                console.error("Failed to request condition approval:", error);
                dispatch(failRequestConditionApproval());
            });
    };
}

export function approveRuleCondition(api, lang, ruleId, source) {
    return (dispatch) => {
        return api
            .then((api) => api.apis.lab.approveRuleCondition)
            .then((method) => method({ id: ruleId, source: source }))
            .then((response) => {
                dispatch(reachGoal("approveRuleConditionSuccess"));
                return dispatch(fetchRule(api, lang, ruleId));
            })
            .catch((error) => {
                dispatch(reachGoal("approveRuleConditionFail"));
                dispatch(addErrorNotification(error, "Failed to approve condition"));
            });
    };
}

export function deleteRuleCondition(api, lang, ruleId, source, state) {
    return (dispatch) => {
        return api
            .then((api) => api.apis.lab.deleteRuleCondition)
            .then((method) => method({ id: ruleId, source: source, state: state }))
            .then((response) => {
                dispatch(reachGoal("deleteRuleConditionSuccess"));
                return dispatch(fetchRule(api, lang, ruleId));
            })
            .catch((error) => {
                dispatch(reachGoal("deleteRuleConditionFailed"));
                dispatch(addErrorNotification(error, "Failed to delete condition"));
            });
    };
}

export function addSegmentExportTag(api, segmentExportId, tag) {
    return (dispatch) => {
        return api
            .then((api) => api.apis.lab.addTag)
            .then((method) => method({ id: segmentExportId, tag: tag }))
            .then((response) => {
                dispatch(reachGoal("addSegmentExportTagSuccess"));
            })
            .catch((error) => {
                dispatch(addErrorNotification(error, "Failed adding tags: export_id = " + segmentExportId));
                dispatch(reachGoal("addSegmentExportTagFailed"));
            });
    };
}

export function deleteSegmentExportTag(api, segmentExportId, tag) {
    return (dispatch) => {
        return api
            .then((api) => api.apis.lab.deleteTag)
            .then((method) => method({ id: segmentExportId, tag: tag }))
            .then((response) => {
                dispatch(reachGoal("deleteSegmentExportTagSuccess"));
            })
            .catch((error) => {
                dispatch(addErrorNotification(error, "Failed deleting tags: export_id = " + segmentExportId));
                dispatch(reachGoal("deleteSegmentExportTagFailed"));
            });
    };
}

export function addManySegmentExportTags(api, segmentId, tags) {
    return (dispatch) => {
        tags.forEach((tag) => {
            dispatch(addSegmentExportTag(api, segmentId, tag));
        });
    };
}

export function deleteManySegmentExportTags(api, segmentId, tags) {
    return (dispatch) => {
        tags.forEach((tag) => {
            dispatch(deleteSegmentExportTag(api, segmentId, tag));
        });
    };
}

export function openAndInitiateExpressionsModal(api, exportId) {
    return (dispatch) => {
        dispatch(reachGoal("openExpressionsModal"));
        dispatch(setExpressionsModalVisibility(true));
        dispatch(fetchSegmentExport(api, exportId));
    };
}

export function openAndInitiateLalModal(api, exportId) {
    return (dispatch) => {
        dispatch(reachGoal("openLalModal"));
        dispatch(setLalModalVisibility(true));
        dispatch(fetchSegmentExport(api, exportId));
    };
}

export const requestStoreExpressions = createAction("request store expressions");
export const receiveStoreExpressions = createAction("receive store expressions");
export const failStoreExpressions = createAction("fail store expressions");

export function storeExpressions(api, exportId, expressions) {
    return (dispatch) => {
        dispatch(requestSegmentExport(exportId));

        return api
            .then((api) => api.apis.lab.putExportExpression)
            .then((method) => method({ id: exportId, expressions: expressions }))
            .then((response) => {
                dispatch(receiveSegmentExport(response.obj));
                dispatch(reachGoal("storeExpressionsSuccess"));
            })
            .catch((error) => {
                dispatch(addErrorNotification(error, "Failed to store expression"));
                dispatch(failSegmentExport());
                dispatch(reachGoal("storeExpressionsFailed"));
            });
    };
}

export const requestStoreLal = createAction("request store lal");
export const receiveStoreLal = createAction("receive store lal");
export const failStoreLal = createAction("fail store lal");

export function storeLal(api, exportId, lal) {
    return (dispatch) => {
        dispatch(requestStoreLal(exportId));

        return api
            .then((api) => api.apis.lab.putLal)
            .then((method) => method({ id: exportId, lal: lal }))
            .then((response) => {
                dispatch(
                    receiveStoreLal({
                        exportId: response.obj.id,
                        storedLal: response.obj.lal,
                    })
                );
                dispatch(reachGoal("storeLal"));
            })
            .catch((error) => {
                dispatch(addErrorNotification(error, "Failed to store lal"));
                dispatch(failStoreLal());
                dispatch(reachGoal("storeLalFailed"));
            });
    };
}

export const requestSegmentExport = createAction("request segment export");
export const receiveSegmentExport = createAction("receive segment export");
export const failSegmentExport = createAction("fail segment export");

export function fetchSegmentExport(api, exportId) {
    return (dispatch) => {
        dispatch(requestSegmentExport(exportId));

        return api
            .then((api) => api.apis.lab.getSegmentExport)
            .then((method) => method({ id: exportId }))
            .then((response) => {
                dispatch(receiveSegmentExport(response.obj));
            })
            .catch((error) => {
                console.error("Unable to fetch segmentExport", error);
                dispatch(failSegmentExport());
            });
    };
}

export const requestDeleteExpressions = createAction("request delete expressions");
export const receiveDeleteExpressions = createAction("receive delete expressions");
export const failDeleteExpressions = createAction("fail delete expressions");

export function deleteExpressions(api, exportId) {
    return (dispatch) => {
        dispatch(requestDeleteExpressions(exportId));

        return api
            .then((api) => api.apis.lab.deleteExportExpression)
            .then((method) => method({ id: exportId }))
            .then((response) => {
                dispatch(receiveDeleteExpressions(response.obj));
                dispatch(reachGoal("deleteExpressionSuccess"));
            })
            .catch((error) => {
                dispatch(failDeleteExpressions());
                dispatch(reachGoal("deleteExpressionFailed"));
            });
    };
}

export const requestDeleteLal = createAction("request delete lal");
export const receiveDeleteLal = createAction("receive delete lal");
export const failDeleteLal = createAction("fail delete lal");

export function deleteLal(api, exportId) {
    return (dispatch) => {
        dispatch(requestDeleteLal(exportId));

        return api
            .then((api) => api.apis.lab.deleteLal)
            .then((method) => method({ id: exportId }))
            .then((response) => {
                dispatch(receiveDeleteLal(response.obj));
                dispatch(reachGoal("deleteLalSuccess"));
            })
            .catch((error) => {
                dispatch(addErrorNotification(error, "Failed to delete lal"));
                dispatch(failDeleteLal());
                dispatch(reachGoal("deleteLalFailed"));
            });
    };
}

export const requestExportInfo = createAction("request export info");
export const receiveExportInfo = createAction("receive export info");
export const failExportInfo = createAction("fail export info");

export const showExportTooltip = createAction("show export tooltip");
export const hideExportTooltip = createAction("hide export tooltip");

export function fetchExportInfo(api, exportId) {
    return (dispatch) => {
        dispatch(requestExportInfo(exportId));

        return api
            .then((api) => api.apis.lab.getSegmentExport)
            .then((method) => method({ id: exportId }))
            .then((response) => {
                dispatch(receiveExportInfo(response.obj));
                dispatch(reachGoal("checkExportInfoSuccess"));
            })
            .catch((error) => {
                dispatch(failExportInfo(exportId));
                dispatch(reachGoal("checkExportInfoFailed"));
            });
    };
}

export const requestSegmentByExportId = createAction("request segment by export id");
export const receiveSegmentByExportId = createAction("receive segment by export id");
export const failSegmentByExportId = createAction("fail segment by export id");

export function fetchSegmentByExportId(api, exportId) {
    return (dispatch) => {
        dispatch(requestSegmentByExportId());

        return api
            .then((api) => api.apis.lab.getSegmentByExportId)
            .then((method) => method({ id: exportId }))
            .then((response) => {
                dispatch(receiveSegmentByExportId(response.obj));
            })
            .catch((error) => {
                dispatch(failSegmentByExportId());
                dispatch(addErrorNotification(error, "Failed to getSegmentInfo"));
            });
    };
}

export const requestSegmentStats = createAction("request segment stats");
export const receiveSegmentStats = createAction("receive segment stats");
export const failSegmentStats = createAction("fail segment stats");


export function fetchRelativeSegmentStats(api, exportId, baseExportId, baseSampleId, baseGroupId) {
    return (dispatch) => {
        dispatch(requestSegmentStats());

        let params = {
            id: exportId,
            base_export_id:baseExportId,
            base_sample_id: baseSampleId,
            base_group_id: baseGroupId,
        }
        return api
            .then((api) => api.apis.lab.getSegmentSiberiaStats)
            .then((method) => method(params))
            .then((response) => {
                dispatch(receiveSegmentStats(response.obj));
                dispatch(reachGoal("fetchSegmentStatsSuccess"));
            })
            .catch((error) => {
                dispatch(addErrorNotification(error, "Failed loading stats"));
                dispatch(failSegmentStats());
                dispatch(reachGoal("fetchSegmentStatsFailed"));
            });
    };
}

export const requestRuleRevisionEstimateStats = createAction("request rule revision stats");
export const receiveRuleRevisionEstimateStats = createAction("receive rule revision stats");
export const failRuleRevisionEstimateStats = createAction("fail rule revision stats");

export function fetchRuleRevisionEstimateStats(api, ruleRevisionId, lang) {
    return (dispatch) => {
        dispatch(requestRuleRevisionEstimateStats(ruleRevisionId));

        return api
            .then((api) => api.apis.lab.getRuleConditionEstimateStatsByRevision)
            .then((method) => method({ revision: ruleRevisionId, lang: lang }))
            .then((response) => {
                dispatch(receiveRuleRevisionEstimateStats(response.obj));
                dispatch(reachGoal("fetchRuleRevisionEstimateStatsSuccess"));
            })
            .catch((error) => {
                dispatch(addErrorNotification(error, "Failed loading stats"));
                dispatch(failRuleRevisionEstimateStats());
                dispatch(reachGoal("fetchRuleRevisionEstimateStatsFailed"));
            });
    };
}

export const requestRuleRevisionEstimateCoverage = createAction("request rule revision coverage");
export const receiveRuleRevisionEstimateCoverage = createAction("receive rule revision coverage");
export const failRuleRevisionEstimateCoverage = createAction("fail rule revision coverage");

export function fetchRuleRevisionEstimateCoverage(api, ruleRevisionId, lang) {
    return (dispatch) => {
        dispatch(requestRuleRevisionEstimateCoverage(ruleRevisionId));

        return api
            .then((api) => api.apis.lab.getRuleConditionEstimateCoverageByRevision)
            .then((method) => method({ revision: ruleRevisionId, lang: lang }))
            .then((response) => {
                dispatch(receiveRuleRevisionEstimateCoverage(response.obj));
                dispatch(reachGoal("fetchRuleRevisionEstimateCoverageSuccess"));
            })
            .catch((error) => {
                dispatch(failRuleRevisionEstimateCoverage());
                dispatch(reachGoal("fetchRuleRevisionEstimateCoverageFailed"));
            });
    };
}

export function updateRuleEstimates(api, ruleId, ruleRevisionId, lang) {
    return (dispatch) => {
        return api
            .then((api) => api.apis.lab.updateRuleEstimateStats)
            .then((method) => method({ rule_id: ruleId }))
            .then((response) => {
                dispatch(addSuccessNotification("Stats update scheduled"));
                dispatch(reachGoal("updateRuleEstimatesSuccess"));
                dispatch(fetchRuleEstimateCoverage(api, ruleId, lang));
                dispatch(fetchRuleRevisionEstimateCoverage(api, ruleRevisionId, lang));
            })
            .catch((error) => {
                dispatch(addErrorNotification(error, "Failed updating stats"));
                dispatch(reachGoal("updateRuleEstimatesFailed"));
            });
    };
}

export const requestRuleEstimateStats = createAction("request rule stats");
export const receiveRuleEstimateStats = createAction("receive rule stats");
export const failRuleEstimateStats = createAction("fail rule stats");

export function fetchRuleEstimateStats(api, ruleId, lang) {
    return (dispatch) => {
        dispatch(requestRuleEstimateStats(ruleId));

        return api
            .then((api) => api.apis.lab.getRuleEstimateStats)
            .then((method) => method({ rule_id: ruleId, lang: lang }))
            .then((response) => {
                dispatch(receiveRuleEstimateStats(response.obj));
                dispatch(reachGoal("fetchRuleEstimateStatsSuccess"));
            })
            .catch((error) => {
                dispatch(addErrorNotification(error, "Failed loading stats"));
                dispatch(failRuleEstimateStats());
                dispatch(reachGoal("fetchRuleEstimateStatsFailed"));
            });
    };
}

export const requestRuleEstimateCoverage = createAction("request rule coverage");
export const receiveRuleEstimateCoverage = createAction("receive rule coverage");
export const failRuleEstimateCoverage = createAction("fail rule coverage");

export function fetchRuleEstimateCoverage(api, ruleId, lang) {
    return (dispatch) => {
        dispatch(requestRuleEstimateCoverage(ruleId));

        return api
            .then((api) => api.apis.lab.getRuleEstimateCoverage)
            .then((method) => method({ rule_id: ruleId, lang: lang }))
            .then((response) => {
                dispatch(receiveRuleEstimateCoverage(response.obj));
                dispatch(reachGoal("fetchRuleEstimateCoverageSuccess"));
            })
            .catch((error) => {
                dispatch(failRuleEstimateCoverage());
                dispatch(reachGoal("fetchRuleEstimateCoverageFailed"));
            });
    };
}

export function deleteExport(api, exportId, segmentId, lang) {
    return (dispatch) => {
        return api
            .then((api) => api.apis.lab.deleteSegmentExport)
            .then((method) => method({ id: exportId, lang: lang }))
            .then((response) => {
                dispatch(fetchSegment(api, lang, segmentId));
            })
            .catch((error) => {
                dispatch(addErrorNotification(error, "Failed to delete export"));
            });
    };
}


export function deleteSegement(api, segmentId, lang) {
    return (dispatch) => {
        return api
            .then((api) => api.apis.lab.deleteSegment)
            .then((method) => method({ id: segmentId, lang: lang }))
            .then((response) => {
                dispatch(fetchSegmentsAndGroups(api, lang));
            })
            .catch((error) => {
                dispatch(addErrorNotification(error, "Failed to delete segment"));
            });
    };
}


export const requestGrantsForAudienceSegment = createAction("request grants for audience segment");
export const receiveGrantsForAudienceSegment = createAction("receive grants for audience segment");
export const failGrantsForAudienceSegment = createAction("fail grants for audience segment");

export function fetchGrantsForAudienceSegment(api, segmentId) {
    return (dispatch) => {
        dispatch(requestGrantsForAudienceSegment(segmentId));

        return api
            .then((api) => api.apis.lab.listGrants)
            .then((method) => method({ segmentId: segmentId }))
            .then((response) => {
                dispatch(receiveGrantsForAudienceSegment({
                    grants: Object.values(response.obj.grants).map((grant) => grant.user_login),
                    segmentId: segmentId,
                }));
            })
            .catch((error) => {
                console.error("Failed to get grants for segmentId=" + segmentId);
                dispatch(failGrantsForAudienceSegment(segmentId));
            });
    };
}


export const invalidateGrantsForManyAudienceSegments = createAction("invalidate grants for many audience segments");

export function fetchGrantsForManyAudienceSegments(api, segmentIds) {
    return (dispatch) => {
        dispatch(invalidateGrantsForManyAudienceSegments());

        segmentIds.forEach((segmentId) => {
            dispatch(fetchGrantsForAudienceSegment(api, segmentId));
        });
    };
}


