import React, { useState, useEffect } from "react";
import PropTypes from "prop-types";
import { connect, useDispatch } from "react-redux";
import { isEmpty } from "lodash";
import { useTranslation } from "react-i18next";
import FontAwesome from "react-fontawesome";
import { Button, Textarea, Spin, Modal } from "@lib/components/lego";

import { getApiState } from "../../selectors";
import {
    addManyGrants,
    deleteGrant,
    deleteManyGrants,
    fetchAudienceSegmentGrants,
    invalidateCreatedGrants,
    invalidateDeletedGrants,
} from "./audienceActions";
import { getAudienceSegmentGrantsState, getCreatedGrantsState } from "./audienceSelectors";

import "./AudienceAccess.scss";
import { Link } from "@yandex-lego/components/Link";

const AudienceAccess = connect((state) => ({
    api: getApiState(state),
    grants: getAudienceSegmentGrantsState(state).value,
    loaded: !getAudienceSegmentGrantsState(state).loading,
    error: getAudienceSegmentGrantsState(state).error,
    newGrants: getCreatedGrantsState(state),
}))((props) => {
    const { api, segmentId, grants, loaded, error, newGrants } = props;

    const [editMode, setEditMode] = useState(false);
    const [grantsList, setGrantsList] = useState([]);
    const [failedGrants, setFailedGrants] = useState([]);
    const [comment, setComment] = useState("");
    const [externalAdvertiserTooltipVisibility, setExternalAdvertiserTooltipVisibility] = useState(false);

    const { t } = useTranslation("segments");
    const dispatch = useDispatch();

    const fetchGrants = (api, segmentId) => dispatch(fetchAudienceSegmentGrants(api, segmentId));

    const editGrants = () => {
        setEditMode(!editMode);
        dispatch(invalidateCreatedGrants());
        dispatch(invalidateDeletedGrants());
    };

    const deleteSegmentGrant = (userLogin) => {
        dispatch(deleteGrant(api, segmentId, userLogin));
        setTimeout(() => fetchGrants(api, segmentId), 500);
    };

    const grantsLoaded = loaded && !error;

    grants.sort(function(a, b) {
        let a_login = a.user_login.toLowerCase();
        let b_login = b.user_login.toLowerCase();

        if (a_login > b_login) {
            return 1;
        } else if (a_login < b_login) {
            return -1;
        } else {
            return 0;
        }
    });

    useEffect(() => {
        dispatch(invalidateCreatedGrants());
    }, []);

    useEffect(() => {
        fetchGrants(api, segmentId);
    }, [segmentId]);

    useEffect(() => {
        if (grants) {
            setGrantsList(grants.map((grant) => grant.user_login));
        }
    }, [grants]);

    useEffect(() => {
        if (newGrants.length !== 0) {
            let failed = Object.keys(newGrants).filter((grant) => newGrants[grant].error === "NOT_AUTHENTICATED");
            setFailedGrants(failed);
            if (failed.length !== 0) {
                setExternalAdvertiserTooltipVisibility(true);
            } else {
                setExternalAdvertiserTooltipVisibility(false);
            }
        }
    }, [newGrants]);

    const renderEditGrants = () => {
        const loginsAsText = grantsList.join("\n");
        const updatedLogins = (currentLogins) => {
            return currentLogins.split(/\s*,\s*|\s+/);
        };

        const saveChanges = () => {
            setEditMode(!editMode);

            let newLogins = grantsList
                .filter((editedLogin) => !grants.map((grant) => grant.user_login).includes(editedLogin))
                .filter((grant) => !isEmpty(grant));

            let oldLogins = grants
                .map((grant) => grant.user_login)
                .filter((grant) => !grantsList.includes(grant))
                .filter((grant) => !isEmpty(grant));

            Promise.all([
                dispatch(addManyGrants(api, segmentId, newLogins, comment)),
                dispatch(deleteManyGrants(api, segmentId, oldLogins)),
            ]).then((done) => {
                setTimeout(() => fetchGrants(api, segmentId), 500);
            });
        };

        const rejectChanges = () => {
            setEditMode(false);
            setGrantsList(grants.map((grant) => grant.user_login));
        };

        return (
            <div className="AudienceAccess-edit-mode">
                <div className="AudienceAccess-textarea-title">{t("audience.grants.loginsList")}</div>
                <Textarea
                    className="AudienceAccess-logins-textarea"
                    size="m"
                    view="default"
                    value={loginsAsText}
                    onChange={(event) => setGrantsList(updatedLogins(event.target.value))}
                />

                <div className="AudienceAccess-textarea-title">
                    {t("audience.grants.commentFor")}
                    <b>{t("audience.grants.new")}</b>
                    {t("audience.grants.grants")}
                </div>
                <Textarea
                    className="AudienceAccess-comment-textarea"
                    size="m"
                    view="default"
                    value={comment}
                    onChange={(event) => setComment(event.target.value)}
                />

                <div className="AudienceAccess-edit-controls">
                    <Button view="action" size="m" onClick={saveChanges} className="AudienceAccess-save-changes-button">
                        {t("audience.grants.saveChanges")}
                    </Button>
                    <Button view="pseudo" size="m" onClick={rejectChanges}>
                        {t("audience.grants.rejectChanges")}
                    </Button>
                </div>
            </div>
        );
    };

    const renderGrantsList = () => {
        return grantsLoaded ? (
            <div>
                <div className="AudienceAccess-logins-table-container">
                    <table className="AudienceAccess-logins-table">
                        <tbody>
                            {grants.map((grant, index) => {
                                let isNew = Object.keys(newGrants).includes(grant.user_login);

                                return (
                                    <tr key={index}>
                                        <td className="AudienceAccess-user-login">
                                            {isNew && <span className="AudienceAccess-new-user-login" />}
                                            <span className="AudienceAccess-user-login-value">{grant.user_login}</span>
                                        </td>
                                        <td className="AudienceAccess-comment">{grant.comment}</td>
                                        <td className="AudienceAccess-delete-grant">
                                            <FontAwesome
                                                name="trash"
                                                onClick={() => deleteSegmentGrant(grant.user_login)}
                                            />
                                        </td>
                                    </tr>
                                );
                            })}
                        </tbody>
                    </table>
                </div>

                <div className="AudienceAccess-edit-controls">
                    <Button
                        view="action"
                        size="m"
                        onClick={editGrants}
                    >
                        {t("audience.grants.edit")}
                    </Button>
                </div>
                <Modal theme="normal" visible={externalAdvertiserTooltipVisibility} onClose={() => setExternalAdvertiserTooltipVisibility(false)}>
                    <div className="advertiser-type-selector-popup">{t("audience.advertiserTypeModal.tooltip.disclaimer")}</div>
                    <div className="advertiser-type-selector-popup">
                        <table className="advertiser-type-selector-popup-table">
                            <tbody>
                            {failedGrants.map((grant, index) => {
                                return (
                                    <tr key={index}>
                                        <td className="AudienceAccess-user-login">
                                            <span className="AudienceAccess-user-login-value">{grant}</span>
                                        </td>
                                    </tr>
                                );
                            })}
                            </tbody>
                        </table>
                    </div>
                    <div className="advertiser-type-selector-popup">{t("audience.advertiserTypeModal.tooltip.start")}<Link tabindex={-1} href="https://docs.yandex-team.ru/crypta/lab/audience-share#1-login-ne-ukazan-na-staffe">{t("audience.advertiserTypeModal.tooltip.link")}</Link>{t("audience.advertiserTypeModal.tooltip.end")}</div>
                </Modal>
            </div>
        ) : (
            <div className="AudienceAccess-logins-loading">
                <Spin progress view="default" size="m" />
            </div>
        );
    };

    return (
        <div className="AudienceAccess-container">
            <div className="AudienceAccess-header">{t("audience.accessHeader")}</div>
            <div className="AudienceAccess-logins">{!editMode ? renderGrantsList() : renderEditGrants()}</div>
        </div>
    );
});

AudienceAccess.propTypes = {
    segmentId: PropTypes.oneOfType([PropTypes.string.isRequired, PropTypes.number.isRequired]),
};

export default AudienceAccess;
