import React, { useState, useEffect } from "react";
import { connect, useDispatch } from "react-redux";
import { useTranslation } from "react-i18next";
import { isUndefined } from "lodash";
import FontAwesome from "react-fontawesome";
import { Modal, TextInput, Button } from "@lib/components/lego";
import Loading from "@lib/components/loading";
import { getSegmentsModalsState } from "../selectors";
import { changeAudienceSegmentName, setAudienceSegmentFeaturesModalVisibility } from "./audienceActions";
import LabModalHeader from "../../utils/LabModalHeader";
import { getApiState } from "../../selectors";
import { getChangedSegmentNameState, getSegmentAudienceState } from "./audienceSelectors";

import Failable from "../../utils/Failable";
import { prettifyNumber } from "./utils";
import AudienceSegmentStatus from "./AudienceSegmentStatus";
import AudienceAccess from "./AudienceAccess";

import "./AudienceSegmentFeatures.scss";

const AudienceSegmentFeatures = connect((state) => ({
    api: getApiState(state),
    visible: getSegmentsModalsState(state).audiencesSegmentFeatures,
    segment: getSegmentAudienceState(state).value,
    loading: getSegmentAudienceState(state).loading,
    error: getSegmentAudienceState(state).error,
    name: getChangedSegmentNameState(state),
}))((props) => {
    const { api, visible, segment, loading, error, name } = props;

    const [segmentName, setSegmentName] = useState("");
    const [editNameMode, setEditNameMode] = useState(false);

    const { t } = useTranslation("segments");
    const dispatch = useDispatch();

    useEffect(() => {
        if (segment) {
            setSegmentName(segment.name);
        }
    }, [segment]);

    const changeSegmentName = () => {
        Promise.resolve(dispatch(changeAudienceSegmentName(api, segment.id, segmentName))).then((result) => {
            setEditNameMode(false);
        });
    };

    useEffect(() => {
        setSegmentName(name.result);
    }, [name.result]);

    const cancelChange = () => {
        setSegmentName(segment.name);
        setEditNameMode(false);
    };

    const close = () => dispatch(setAudienceSegmentFeaturesModalVisibility(false));

    const renderSegmentName = () => {
        return !editNameMode ? (
            <span className="AudienceSegmentFeatures-segment-name">
                {segmentName}
                <span className="AudienceSegmentFeatures-edit-name-button" onClick={() => setEditNameMode(true)}>
                    <FontAwesome name="pencil" />
                </span>
            </span>
        ) : (
            <div className="AudienceSegmentFeatures-edit-name-mode">
                <TextInput
                    className="AudienceSegmentFeatures-segment-name-input"
                    size="m"
                    view="default"
                    value={segmentName}
                    disabled={name.loading}
                    hasClear
                    onChange={(event) => setSegmentName(event.target.value)}
                />
                <div className="AudienceSegmentFeatures-edit-name-actions">
                    <Button
                        size="s"
                        view="action"
                        onClick={changeSegmentName}
                        disabled={name.loading}
                        className="AudienceSegmentFeatures-edit-name-save-button"
                    >
                        {t("audience.name.save")}
                    </Button>
                    <Button size="s" view="default" onClick={cancelChange} disabled={name.loading}>
                        {t("audience.name.cancel")}
                    </Button>
                </div>
            </div>
        );
    };

    return (
        <Modal theme="normal" onClose={close} visible={visible}>
            <LabModalHeader
                title={t("audience.settingsHeader")}
                info={segment ? segment.id : ""}
                hasClose
                onClose={close}
            />
            <div className="AudienceSegmentFeatures-content">
                <Failable failed={error} errorMessage={t("audience.failedLoading")}>
                    <Loading loaded={!loading && !isUndefined(segment)} size="m">
                        {segment && (
                            <table className="AudienceSegmentFeatures-table">
                                <tbody>
                                    <tr>
                                        <td className="AudienceSegmentFeatures-label">
                                            {t("audience.audiencesSegmentName")}
                                        </td>
                                        <td className="AudienceSegmentFeatures-value">{renderSegmentName()}</td>
                                    </tr>
                                    <tr>
                                        <td className="AudienceSegmentFeatures-label">{t("audience.statusLabel")}</td>
                                        <td className="AudienceSegmentFeatures-value">
                                            <AudienceSegmentStatus status={segment.status}>
                                                {t("audience.status." + segment.status)}
                                            </AudienceSegmentStatus>
                                        </td>
                                    </tr>
                                    <tr>
                                        <td className="AudienceSegmentFeatures-label">{t("audience.coverage")}</td>
                                        <td className="AudienceSegmentFeatures-value">
                                            {prettifyNumber(segment.cookies_matched_quantity)}
                                        </td>
                                    </tr>
                                    <tr>
                                        <td className="AudienceSegmentFeatures-label">{t("audience.createTime")}</td>
                                        <td className="AudienceSegmentFeatures-value">{segment.create_time}</td>
                                    </tr>
                                </tbody>
                            </table>
                        )}
                    </Loading>
                </Failable>

                {segment && <AudienceAccess segmentId={segment.id} />}
            </div>
        </Modal>
    );
});

export default AudienceSegmentFeatures;
