import { createAction } from "redux-actions";
import { addErrorNotification, addSuccessNotification } from "../../utils/notifications/notificationsState";

export const setAudienceSegmentFeaturesModalVisibility = createAction("set audiences features modal visibility");

export const requestAudienceSegment = createAction("request audiences segment");
export const receiveAudienceSegment = createAction("receive audiences segment");
export const failAudienceSegment = createAction("fail audiences segment");

export const requestGrants = createAction("request segment grants");
export const receiveGrants = createAction("receive segment grants");
export const failGrants = createAction("fail segment grants");

export function fetchAudienceSegment(api, segmentId) {
    return (dispatch) => {
        dispatch(requestAudienceSegment);

        return api
            .then((api) => api.apis.lab.getAudienceSegment({ segmentId: segmentId }))
            .then((response) => {
                dispatch(receiveAudienceSegment(response.obj));
            })
            .catch((error) => {
                console.error("Unable to fetch Audiences segment segmentId=" + segmentId + "\n" + error);
                dispatch(failAudienceSegment());
            });
    };
}

export function fetchAudienceSegmentGrants(api, segmentId) {
    return (dispatch) => {
        dispatch(requestGrants());

        return api
            .then((api) => api.apis.lab.listGrants)
            .then((method) => method({ segmentId: segmentId }))
            .then((response) => {
                dispatch(receiveGrants(response.obj.grants));
            })
            .catch((error) => {
                console.error("Failed to get grants for segmentId=" + segmentId);
                dispatch(failGrants());
            });
    };
}

export const requestCreateGrant = createAction("request create grant");
export const receiveCreateGrant = createAction("receive create grant");
export const failCreateGrant = createAction("fail create grant");
export const invalidateCreatedGrants = createAction("invalidate created grants");
export const receiveCreateManyGrants = createAction("receive create many grants");
export const failCreateManyGrants = createAction("receive create many grants");

export function addManyGrants(api, segmentId, logins, comment) {
    return (dispatch) => {
        dispatch(invalidateCreatedGrants());

        return api
            .then((api) => api.apis.lab.createGrants)
            .then((method) => {
                logins.forEach(login => dispatch(requestCreateGrant(login)));
                 return method({
                    segmentId: segmentId,
                    logins: logins,
                    comment: comment,
                });
            })
            .then((response) => {
                response.obj.forEach(grant => {
                    if (grant.receivedLogin) {
                        dispatch(receiveCreateGrant({
                            requestedLogin: grant.requestedLogin,
                            receivedLogin: grant.receivedLogin,
                        }));

                        dispatch(addSuccessNotification(`Grant added: id=${segmentId}; login=${grant.receivedLogin}`));
                    } else {
                        dispatch(
                            failCreateGrant({
                                requestedLogin: grant.requestedLogin,
                                status: grant.error.reason,
                            })
                        );
                        dispatch(addErrorNotification({
                            status: grant.error.reason,
                            obj: {
                                message: grant.error.message,
                            }
                        }, `Доступ логину ${grant.requestedLogin} не выдан.`));
                    }
                });
                dispatch(receiveCreateManyGrants());
            })
            .catch((error) => {
                dispatch(addErrorNotification(error, ""));
                dispatch(failCreateManyGrants());
            });
    };
}

export const requestDeleteGrant = createAction("request delete grant");
export const receiveDeleteGrant = createAction("receive delete grant");
export const failDeleteGrant = createAction("fail delete grant");

export function deleteGrant(api, segmentId, userLogin) {
    return (dispatch) => {
        dispatch(requestDeleteGrant(userLogin));

        return api
            .then((api) => api.apis.lab.deleteGrant)
            .then((method) => method({ segmentId: segmentId, userLogin: userLogin }))
            .then((response) => {
                dispatch(
                    receiveDeleteGrant({
                        userLogin: userLogin,
                        status: response.obj.success,
                    })
                );

                dispatch(addSuccessNotification(`Grant removed: id=${segmentId}; login=${userLogin}`));
            })
            .catch((error) => {
                console.error('Unable to delete grant for user login"', userLogin, '"');
                dispatch(failDeleteGrant(userLogin));
            });
    };
}

export const invalidateDeletedGrants = createAction("invalidate deleted grants");

export function deleteManyGrants(api, segmentId, logins) {
    return (dispatch) => {
        dispatch(invalidateDeletedGrants());

        logins.forEach((login) => {
            dispatch(deleteGrant(api, segmentId, login));
        });
    };
}

export const requestChangeSegmentName = createAction("request change segment name");
export const receiveChangeSegmentName = createAction("receive change segment name");
export const failChangeSegmentName = createAction("fail change segment name");

export function changeAudienceSegmentName(api, segmentId, segmentName) {
    return (dispatch) => {
        dispatch(requestChangeSegmentName());

        return api
            .then((api) => api.apis.lab.updateAudienceName)
            .then((method) => method({ segmentId: segmentId, name: segmentName }))
            .then((response) => {
                dispatch(receiveChangeSegmentName(response.obj.segment.name));
            })
            .catch((error) => {
                console.error("Failed to update segment name, id=" + segmentId);
                dispatch(failChangeSegmentName());
            });
    };
}
