import React, { useEffect, useState } from "react";
import { connect, useDispatch } from "react-redux";
import { isEmpty } from "lodash";
import { Button, Modal, Textarea } from "@lib/components/lego";
import { getActiveIdsState, getActiveSegmentInfoState } from "../selectors";
import { addManySegmentExportTags, deleteManySegmentExportTags, fetchSegment } from "../actions";
import { getApiState } from "../../selectors";
import { useTranslation } from "react-i18next";
import LabModalHeader from "../../utils/LabModalHeader";

import "./ExportTagsModal.scss";

const ExportTagsModal = connect((state) => ({
    api: getApiState(state),
    segmentExportId: state.segments.exportSegmentId,
    activeSegmentId: getActiveIdsState(state).segment,
    activeSegmentInfo: getActiveSegmentInfoState(state),
}))((props) => {
    const { api, visible, segmentExportId, segmentExport, activeSegmentId, activeSegmentInfo, onClose } = props;

    const dispatch = useDispatch();
    const { t, i18n } = useTranslation("segments");

    const [tagsList, setTagsList] = useState(segmentExport.tags);

    const addTags = (values) => dispatch(addManySegmentExportTags(api, segmentExportId, values));
    const deleteTags = (values) => dispatch(deleteManySegmentExportTags(api, segmentExportId, values));

    const tags = segmentExport.tags;

    useEffect(() => {
        setTagsList(segmentExport.tags);
    }, [segmentExport.tags]);

    const close = () => onClose();

    const tagsAsText = tagsList.join("\n");
    const updatedTags = (currentTags) => {
        return currentTags.split(/\s*,\s*|\s+/);
    };

    const saveChanges = () => {
        let newTags = tagsList.filter((tag) => !tags.includes(tag) && !isEmpty(tag));
        let oldTags = tags.filter((tag) => !tagsList.includes(tag));

        Promise.all([addTags(newTags), deleteTags(oldTags)]).then((done) => {
            setTimeout(() => dispatch(fetchSegment(api, i18n.language, activeSegmentId)));
        });
    };

    const rejectChanges = () => {
        setTagsList(tags);
        close();
    };

    const loading = activeSegmentInfo.loading;

    return (
        <Modal visible={visible} onClose={close} theme="normal">
            <LabModalHeader title={t("tags.header")} hasClose onClose={close} info={segmentExportId} />
            <div className="ExportTagsModal-content">
                <Textarea
                    className="ExportTagsModal-textarea"
                    size="m"
                    view="default"
                    value={tagsAsText}
                    disabled={loading}
                    onChange={(event) => setTagsList(updatedTags(event.target.value))}
                    placeholder={t("tags.placeholder")}
                />

                <div className="ExportTagsModal-actions">
                    <Button
                        view="action"
                        size="s"
                        onClick={saveChanges}
                        disabled={loading}
                        className="ExportTagsModal-actions-save-button"
                    >
                        {t("tags.save")}
                    </Button>
                    <Button view="default" size="s" onClick={rejectChanges} disabled={loading}>
                        {t("tags.cancel")}
                    </Button>
                </div>
            </div>
        </Modal>
    );
});

export default ExportTagsModal;
