import React, {useCallback, useEffect, useState} from "react";
import NumericLabel from "react-pretty-numbers";
import FontAwesome from "react-fontawesome";
import { connect } from "react-redux";
import { indexOf as _indexOf, isEmpty } from "lodash";
import EmbeddableCopyable from "@lib/components/embeddable";
import { simpleFormatNumber } from "@lib/utils/numbers";

import State from "./State";
import SimpleDate from "../../utils/SimpleDate";
import {
    createSegmentExportRule,
    fetchRelativeSegmentStats,
    openAndInitiateExpressionsModal,
    openAndInitiateLalModal,
    setActiveSegmentExportId,
    setActiveSegmentExportTags,
    showRuleModal,
    showSegmentStatsModal,
} from "../actions";
import {
    fetchAudienceSegment,
    fetchAudienceSegmentGrants,
    setAudienceSegmentFeaturesModalVisibility,
} from "../audience/audienceActions";
import {
    getAccessState,
    getActiveIdsState,
    getActiveSegmentInfoState,
    getApiState,
    getSegmentsModalsState,
} from "../../selectors";
import {getAudienceSegmentGrantsState} from "../audience/audienceSelectors";
import DeleteExportDialog from "./delete-export-dialog";

import directPic from "../../images/direct.svg";
import Id from "../../utils/Id";
import { Button, Spin, TooltipStateful } from "@lib/components/lego";
import ExportTags from "./ExportTags";
import { useTranslation } from "react-i18next";
import Link from "@lib/components/link";
import LabLink from "../../utils/LabLink";
import ExportTagsModal from "./ExportTagsModal";
import { allowedToEditSegment } from "../../utils/roles";
import { pushQuery } from "../../actions";
import RuleStatusIcon from "../rule/RuleStatusIcon";

import "../Segments.scss";
const COVERAGE_GRAPH_LINK = "https://grafana.yandex-team.ru/dashboard/db/crypta-analytics-segments";
const DIRECT_USAGE_GRAPH_LINK = "https://grafana.yandex-team.ru/dashboard/db/crypta-segments-usage-in-direct";
const YT_FIELD_NAME = {
    174: "exact_socdem",
    216: "heuristic_segments",
    217: "probabilistic_segments",
    281: "marketing_segments",
    543: "exact_socdem",
    544: "lal_common",
    545: "lal_private",
    546: "lal_internal",
    547: "heuristic_common",
    548: "heuristic_private",
    549: "heuristic_internal",
    557: "audience_segments",
    569: "exact_socdem",
    595: "affinitive_site_ids",
    601: "longterm_interests",
    602: "shortterm_interests",
    614: "exact_socdem",
};

// TODO: make distinct component for these buttons
const BUTTON_SIZE = "s";
const BUTTON_VIEW = "clear";
const BUTTON_THEME = "pseudo";

function isAudienceSegment(keywordId) {
    return keywordId === 557;
}

function getYtField(segment, segmentExport) {
    return segmentExport.keywordId === 217 && segment.type === "INTEREST" // FIXME: what is this?
        ? "interests_composite"
        : YT_FIELD_NAME[segmentExport.keywordId];
}

const BASE_URL = window.location.origin + "/segments";
const SEARCH_URL = "https://search.crypta.yandex-team.ru/?query=";

const SegmentExport = connect((state) => ({
    api: getApiState(state),
    isAdmin: getAccessState(state).Admin,
    isAudienceSegmentSharer: getAccessState(state).AudienceSegmentSharer,
    notAllowedSegments: getAccessState(state).RestrictedSegments,
    active: getActiveIdsState(state),
    showExportTagsModal: getSegmentsModalsState(state).exportTags,
    segment: getActiveSegmentInfoState(state).value,
    grants: getAudienceSegmentGrantsState(state),
}))((props) => {
    const { api, isAdmin, isAudienceSegmentSharer, notAllowedSegments, active, segment, segmentExport, grants, dispatch } =
        props;

    const [tooltipVisible, setTooltipVisibility] = useState(false);
    const [tagsModalVisible, setTagsModalVisibility] = useState(false);
    const [deleteExportDialogVisible, setDeleteExportDialogVisibility] = useState(false);

    const { t, i18n } = useTranslation(["segments", "segmentExport"]);
    const lang = i18n.language;

    useEffect(() => {
        if (isAudienceSegment(segmentExport.keywordId)) {
            dispatch(fetchAudienceSegmentGrants(api, segmentExport.segmentId));
        }
    }, []);

    const openDeleteExportDialog = useCallback(() => {
        setDeleteExportDialogVisibility(true);
    }, []);


    const handleClick = () => {
        const newActiveExport = segmentExport.id !== active.export ? segmentExport.id : undefined;
        const newActive = { ...active, export: newActiveExport };
        dispatch(pushQuery(newActive));
    };

    const handleClickAndStopPropagation = (e) => {
        const newActive = { ...active, export: segmentExport.id };
        dispatch(pushQuery(newActive));
        e.stopPropagation(); // There must be a better solution
    };

    const handleEditTags = (exportId) => {
        dispatch(setActiveSegmentExportId(exportId));
        dispatch(setActiveSegmentExportTags(segmentExport.tags));
        setTagsModalVisibility(true);
    };

    const canCreateAndEditExport = () => {
        return isAdmin || (allowedToEditSegment(segment) && !notAllowedSegments);
    };

    const createDefaultSegmentExportRule = () => {
        dispatch(createSegmentExportRule(api, lang, segment.id, segmentExport.id, segment.name));
    };

    const openRuleWithActualCondition = () => {
        dispatch(showRuleModal(segmentExport.ruleId));
    };

    const handleShowAndRequestStats = () => {
        dispatch(showSegmentStatsModal(segmentExport));
        dispatch(fetchRelativeSegmentStats(api, segmentExport.id));
    };

    const fetchAndShowAudiencesSegmentFeatures = () => {
        dispatch(setAudienceSegmentFeaturesModalVisibility(true));
        dispatch(fetchAudienceSegment(api, segmentExport.segmentId));
    };

    const numericParams = {
        shortFormatMinValue: 1000,
        shortFormat: true,
        cssClass: ["count"],
    };

    const isActive = segmentExport.id === active.export;

    const labelStyle = { fontWeight: "bolder" };
    const idStyle = { fontFamily: 'Consolas, Monaco, "Ubuntu Mono", Courier, monospace', color: "#666" };

    const renderNoData = () => {
        return <div className="segment-info-no-data">-</div>;
    };

    const LalButtonVisible = () => {
        if (![557, 546].includes(segmentExport.keywordId)) {
            return { visible: false, message: "card.export.lal.wrongKeyword" };
        }
        if (segmentExport.ruleId) {
            return { visible: true, message: "card.export.lal.visible" };
        } else {
            for (const exp of segment.exports.exports) {
                if (!isEmpty(exp.ruleId)) {
                    return { visible: true, message: "card.export.lal.visible" };
                }
            }
        }
        return { visible: false, message: "card.export.lal.noRule" };
    };

    let lalButton = LalButtonVisible();

    const valueToEmbed = (
        <table style={{ width: "300px" }}>
            <tbody>
                <tr>
                    <td style={labelStyle}>Export ID</td>
                    <td style={idStyle}>
                        <a href={`${BASE_URL}?segment=${segment.id}&export=${segmentExport.id}`} target="_blank">
                            {segmentExport.id}
                        </a>
                    </td>
                </tr>
                <tr>
                    <td style={labelStyle}>Keyword ID</td>
                    <td style={idStyle}>
                        <a href={`${SEARCH_URL}keyword ${segmentExport.keywordId}`} target="_blank">
                            {segmentExport.keywordId}
                        </a>
                    </td>
                </tr>
                <tr>
                    <td style={labelStyle}>Segment ID</td>
                    <td style={idStyle}>{segmentExport.segmentId}</td>
                </tr>
                <tr>
                    <td style={labelStyle}>Coverage</td>
                    <td>
                        <div>
                            {segmentExport.coverages.profiles.value
                                ? simpleFormatNumber(segmentExport.coverages.profiles.value)
                                : "-"}{" "}
                            / 1 day
                        </div>
                        <div>
                            {segmentExport.coverages.bigb.value
                                ? simpleFormatNumber(segmentExport.coverages.bigb.value)
                                : "-"}{" "}
                            / 14 days
                        </div>
                    </td>
                </tr>
            </tbody>
        </table>
    );

    const spinDefault =
        !isEmpty(segmentExport.lal) && segmentExport.state === "CREATED" ? (
            <Spin className="Spinner" view="default" progress size="xxs" />
        ) : (
            ""
        );

    const uploadingToTheAudienceBy =
        segmentExport.exportTypeId === "yandexuid" ? (
            ""
        ) : (
            <tr>
                <td className="SegmentExport-label">{t("segments:card.export.idType")}</td>
                <td className="SegmentExport-value">
                    <Id value={segmentExport.exportTypeId} copyable={true} />
                </td>
            </tr>
        );

    return <div className={`SegmentExport${isActive ? " SegmentExport-active" : ""}`} onClick={handleClick}>
        <div className="SegmentExport-header">
            <span className="SegmentExport-title">
                <LabLink search={"segment=" + segment.id + "&export=" + segmentExport.id} />
                {t("segments:card.export." + segmentExport.type)}
            </span>
            <span className="SegmentExport-right-part">
                <Id
                    className="SegmentExport-id"
                    value={segmentExport.id}
                    copyable={true}
                    onClick={handleClickAndStopPropagation}
                />

                <EmbeddableCopyable
                    header={segment.name}
                    description={segment.description}
                    className="SegmentExport-embeddable"
                >
                    {valueToEmbed}
                </EmbeddableCopyable>
            </span>

            {canCreateAndEditExport() &&
            (<span>
                <Button className="SegmentExport-trash"
                    view="clear"
                    size="m"
                    onClick={openDeleteExportDialog}
                >
                    <FontAwesome name="trash"/>
                </Button>
            </span>)}
        </div>

        {deleteExportDialogVisible && (
            <DeleteExportDialog
                visible={deleteExportDialogVisible}
                segmentExport={segmentExport}
                grants={grants}
                onClose={() => setDeleteExportDialogVisibility(false)}
            />
        )}
        {tagsModalVisible && (
            <ExportTagsModal
                visible={tagsModalVisible}
                segmentExport={segmentExport}
                onClose={() => setTagsModalVisibility(false)}
            />
        )}

        <table className="SegmentExport-table">
            <tbody>
                <tr>
                    <td className="SegmentExport-label">{t("segments:card.export.state")}</td>
                    <td className="SegmentExport-value">
                        <span onClick={handleClickAndStopPropagation}>
                            <State segmentId={segment.id} segmentExport={segmentExport} isAdmin={isAdmin} canEdit={canCreateAndEditExport()}/>
                        </span>
                    </td>
                </tr>
                <tr>
                    <td className="SegmentExport-label">{t("segments:card.export.ytField")}</td>
                    <td className="SegmentExport-value">{getYtField(segment, segmentExport) || renderNoData()}</td>
                </tr>
                <tr>
                    <td className="SegmentExport-label">Keyword ID</td>
                    <td className="SegmentExport-value">
                        <Id value={segmentExport.keywordId} copyable={true} />
                    </td>
                </tr>
                <tr>
                    <td className="SegmentExport-label">Segment ID</td>
                    <td className="SegmentExport-value">
                        <Id value={segmentExport.segmentId} copyable={true} />
                    </td>
                </tr>
                <tr>
                    <td className="SegmentExport-label">{t("segments:card.export.coverage")}</td>
                    <td className="SegmentExport-value">
                        {segmentExport.coverages.bigb.value ? (
                            <div>
                                <NumericLabel key="value" params={numericParams}>
                                    {segmentExport.coverages.bigb.value}
                                </NumericLabel>
                                <SimpleDate
                                    key="timestamp"
                                    className="segment-timestamp"
                                    lang={lang}
                                    value={segmentExport.coverages.bigb.timestamp}
                                />
                            </div>
                        ) : (
                            renderNoData()
                        )}
                    </td>
                </tr>
                {uploadingToTheAudienceBy}

                <tr>
                    <td className="SegmentExport-label">
                        <span className="EditExportAttributes-left">
                            <span>{t("segments:card.export.tags")}</span>
                            <span className="EditExportAttributes-button">
                                <Button
                                    view="clear"
                                    theme="link"
                                    size="s"
                                    onClick={() => handleEditTags(segmentExport.id)}
                                >
                                    <FontAwesome name="pencil" />
                                </Button>
                            </span>
                        </span>
                    </td>
                    <td className="SegmentExport-value">
                        <ExportTags exportId={segmentExport.id} segmentExport={segmentExport} />
                    </td>
                </tr>
            </tbody>
        </table>

        <div className="SegmentExport-menu">
            <span className="SegmentExport-menu-left">
                <span onClick={handleClickAndStopPropagation}>
                    <TooltipStateful size="s" view="default" content={t("segments:card.export.stats")} hasTail>
                        <Button
                            view={BUTTON_VIEW}
                            theme={BUTTON_THEME}
                            size={BUTTON_SIZE}
                            onClick={handleShowAndRequestStats}
                            onMouseOver={() => setTooltipVisibility(true)}
                            onMouseLeave={() => setTooltipVisibility(false)}
                        >
                            <FontAwesome name="pie-chart" />
                        </Button>
                    </TooltipStateful>
                    {isEmpty(segmentExport.ruleId) && canCreateAndEditExport() ? (
                        <TooltipStateful size="s" view="default" content={t("segments:card.export.addRule")} hasTail>
                            <Button
                                view={BUTTON_VIEW}
                                theme={BUTTON_THEME}
                                size={BUTTON_SIZE}
                                onClick={createDefaultSegmentExportRule}
                            >
                                <FontAwesome name="plus" />
                            </Button>
                        </TooltipStateful>
                    ) : (
                        <TooltipStateful size="s" view="default" content={t("segments:card.export.rule")} hasTail>
                            <Button
                                view={BUTTON_VIEW}
                                theme={BUTTON_THEME}
                                size={BUTTON_SIZE}
                                onClick={openRuleWithActualCondition}
                                disabled={isEmpty(segmentExport.ruleId) && !canCreateAndEditExport()}
                            >
                                <FontAwesome name="th-list" />
                                {!isEmpty(segmentExport.ruleId) && (
                                    <RuleStatusIcon error={segmentExport.ruleHasErrors} />
                                )}
                            </Button>
                        </TooltipStateful>
                    )}
                    <TooltipStateful size="s" view="default" content={t("segments:card.export.expression")} hasTail>
                        <Button
                            view={BUTTON_VIEW}
                            theme={BUTTON_THEME}
                            size={BUTTON_SIZE}
                            disabled={false}
                            onClick={() => dispatch(openAndInitiateExpressionsModal(api, segmentExport.id))}
                        >
                            E
                        </Button>
                    </TooltipStateful>
                    <Link
                        href={`${COVERAGE_GRAPH_LINK}?var-Keyword=${segmentExport.keywordId}&var-Segment=${segmentExport.segmentId}`}
                        target="blank"
                    >
                        <TooltipStateful size="s" view="default" content={t("segments:card.export.coverage")} hasTail>
                            <Button view={BUTTON_VIEW} theme={BUTTON_THEME} size={BUTTON_SIZE}>
                                <FontAwesome name="area-chart" />
                            </Button>
                        </TooltipStateful>
                    </Link>
                    {_indexOf([174, 543, 547, 601, 602, 614], segmentExport.keywordId) !== -1 && (
                        <Link
                            href={`${DIRECT_USAGE_GRAPH_LINK}?var-Keyword=${segmentExport.keywordId}&var-Segment=${segmentExport.segmentId}`}
                            target="blank"
                        >
                            <TooltipStateful size="s" view="default" content={t("segments:card.export.direct")} hasTail>
                                <Button view={BUTTON_VIEW} theme={BUTTON_THEME} size={BUTTON_SIZE}>
                                    <img className="pic" alt="" src={directPic} />
                                </Button>
                            </TooltipStateful>
                        </Link>
                    )}
                    <TooltipStateful size="s" view="default" content={t(lalButton.message)} hasTail>
                        <span>
                            <Button
                                view={BUTTON_VIEW}
                                theme={BUTTON_THEME}
                                size={BUTTON_SIZE}
                                disabled={!lalButton.visible}
                                onClick={() => dispatch(openAndInitiateLalModal(api, segmentExport.id))}
                                addonAfter={spinDefault}
                            >
                                LAL
                            </Button>
                        </span>
                    </TooltipStateful>
                </span>
            </span>
            <span className="SegmentExport-menu-right">
                {(canCreateAndEditExport() || isAudienceSegmentSharer) &&
                    isAudienceSegment(segmentExport.keywordId) && (
                        <Button
                            size="s"
                            view="default"
                            onClick={fetchAndShowAudiencesSegmentFeatures}
                            title={t("segments:audience.settingsButtonHeader")}
                        >
                            <span className="normal-screen-icon">{t("segments:audience.settingsButtonHeader")}</span>
                            <span className="small-screen-icon">
                                <FontAwesome name="users" />
                            </span>
                        </Button>
                    )}
            </span>
        </div>
    </div>
});

export default SegmentExport;
