import React, {useCallback, useState} from "react";
import { connect } from "react-redux";
import { useTranslation } from "react-i18next";
import { isEmpty, sortBy as _sortBy } from "lodash";
import FontAwesome from "react-fontawesome";
import { Button } from "@lib/components/lego";
import Link from "@lib/components/link";
import SegmentExport from "./SegmentExport";
import User from "../../utils/User";
import Id from "../../utils/Id";
import { getAccessState, getApiState, getSegmentsModalsState, getShowDeletedState } from "../../selectors";
import AudienceSegmentFeatures from "../audience/AudienceSegmentFeatures";
import CreateSegmentExportModal from "../export/CreateSegmentExportModal";
import { setCreateAdminExportModalVisibility, setCreateUserExportModalVisibility } from "../export/exportActions";
import CreateAdminSegmentExportModal from "../export/CreateAdminSegmentExportModal";
import EditSegmentDialog from "./edit-segment-dialog";
import DeleteSegmentDialog from "./delete-segment-dialog";

import LabLink from "../../utils/LabLink";

import { getActiveSegmentInfoState } from "../selectors";
import TankerModal from "./tanker";
import {
    addResponsible,
    addStakeholder,
    removeResponsible,
    removeStakeholder,
    setTankerModalVisibility,
} from "../actions";

import "./SegmentInfo.scss";
import { allowedToEditSegment } from "../../utils/roles";
import SegmentUsers from "./SegmentUsers";

const SegmentInfo = connect((state) => ({
    api: getApiState(state),
    showAudiencesSegmentFeatures: getSegmentsModalsState(state).audiencesSegmentFeatures,
    segmentInfo: getActiveSegmentInfoState(state).value,
    loading: getActiveSegmentInfoState(state).loading,
    isAdmin: getAccessState(state).Admin,
    isAudienceSegmentSharer: getAccessState(state).AudienceSegmentSharer,
    tankerModalVisible: getSegmentsModalsState(state).tanker,
    showDeleted: getShowDeletedState(state),
}))((props) => {
    const { api, segmentInfo, loading, showAudiencesSegmentFeatures, isAdmin, isAudienceSegmentSharer, tankerModalVisible, showDeleted, dispatch } = props;

    const [editSegmentDialogVisible, setEditSegmentDialogVisibility] = useState(false);
    const [deleteSegmentDialogVisible, setDeleteSegmentDialogVisibility] = useState(false);

    const { t, i18n } = useTranslation("segments");
    const language = i18n.language;

    const hiddenStates = (isAdmin || isAudienceSegmentSharer) ? []: ["DEPRECATED", "DISABLED"];

    const exports = !isEmpty(segmentInfo) && segmentInfo.exports ? _sortBy(segmentInfo.exports.exports.filter(each => !hiddenStates.includes(each.state) && (each.state !== "DELETED" || showDeleted)), "type") : [];

    const openEditSegmentDialog = useCallback(() => {
        setEditSegmentDialogVisibility(true);
    }, [segmentInfo, showAudiencesSegmentFeatures]);

    const openDeleteSegmentDialog = useCallback(() => {
        setDeleteSegmentDialogVisibility(true);
    }, []);


    const canEditInfo = isAdmin || allowedToEditSegment(segmentInfo);


    const editButton = canEditInfo && (
        <span>
            <Button view="pseudo" size="s" onClick={openEditSegmentDialog}>
                <FontAwesome name="pencil" />
            </Button>
        </span>
    );

    const deleteButton = canEditInfo && (
        <span>
            <Button view="pseudo" size="s" onClick={openDeleteSegmentDialog}>
                <FontAwesome name="trash" />
            </Button>
        </span>
    );


    const addResponsibleFunc = (responsibleLogin) => {
        return dispatch(addResponsible(api, language, segmentInfo.id, responsibleLogin));
    };

    const removeResponsibleFunc = (responsibleLogin) => {
        return dispatch(removeResponsible(api, language, segmentInfo.id, responsibleLogin));
    };

    const addStakeholderFunc = (stakeholderLogin) => {
        return dispatch(addStakeholder(api, language, segmentInfo.id, stakeholderLogin));
    };

    const removeStakeholderFunc = (stakeholderLogin) => {
        return dispatch(removeStakeholder(api, language, segmentInfo.id, stakeholderLogin));
    };

    return (
        !isEmpty(segmentInfo) &&
        segmentInfo.id.startsWith("segment") && (
            <div className="SegmentInfo">
                <div className={loading ? "SegmentInfo-fade-while-loading" : ""}>
                    <div className="SegmentInfo-header">
                        <span>
                            <LabLink search={"segment=" + segmentInfo.id} />
                            <span className="SegmentInfo-header-name">{segmentInfo.name}</span>
                        </span>
                        <span className="SegmentInfo-header-right">
                            {editButton}
                            {deleteButton}
                        </span>
                    </div>

                    {editSegmentDialogVisible && (
                        <EditSegmentDialog
                            visible={editSegmentDialogVisible}
                            segment={segmentInfo}
                            onClose={() => setEditSegmentDialogVisibility(false)}
                        />
                    )}

                    {deleteSegmentDialogVisible && (
                        <DeleteSegmentDialog
                            visible={deleteSegmentDialogVisible}
                            exports={exports}
                            onClose={() => setDeleteSegmentDialogVisibility(false)}
                        />
                    )}

                    <div className="SegmentInfo-description">{segmentInfo.description}</div>

                    {isAdmin && (
                        <div
                            className="SegmentInfo-tanker-button"
                            onClick={() => dispatch(setTankerModalVisibility(true))}
                        >
                            {t("tanker.showDialog")}
                        </div>
                    )}

                    <div className="SegmentInfo-attributes">
                        <div className="SegmentInfo-attribute">
                            <span className="SegmentInfo-attribute-name">ID</span>
                            <span className="SegmentInfo-attribute-value">
                                <Id value={segmentInfo.id} copyable={true} />
                            </span>
                        </div>
                        <div className="SegmentInfo-attribute">
                            <span className="SegmentInfo-attribute-name">{t("card.scope")}</span>
                            <span className="SegmentInfo-attribute-value">{segmentInfo.scope}</span>
                        </div>
                        <div className="SegmentInfo-attribute">
                            <span className="SegmentInfo-attribute-name">ST</span>
                            <span className="SegmentInfo-attribute-value">
                                {segmentInfo.tickets &&
                                    segmentInfo.tickets.map((ticket, i) => (
                                        <div key={i}>
                                            <Link href={`https://st.yandex-team.ru/${ticket}`} target="_blank">
                                                {ticket}
                                            </Link>{" "}
                                        </div>
                                    ))}
                            </span>
                        </div>
                        <div className="SegmentInfo-attribute">
                            <span className="SegmentInfo-attribute-name">{t("card.author")}</span>
                            <span className="SegmentInfo-attribute-value">
                                {segmentInfo.author && <User login={segmentInfo.author} />}
                            </span>
                        </div>

                        <div className="SegmentInfo-attribute">
                            <span className="SegmentInfo-attribute-name">{t("card.responsibles")}</span>
                            <span className="SegmentInfo-attribute-value">
                                <SegmentUsers
                                    segment={segmentInfo.id}
                                    allowedToEdit={canEditInfo}
                                    users={segmentInfo.responsibles}
                                    actionLabel={t("card.addResponsible")}
                                    addUserFunc={addResponsibleFunc}
                                    removeUserFunc={removeResponsibleFunc}
                                    removeActionLabel={t("card.removeFromResponsibles")}
                                />
                            </span>
                        </div>

                        <div className="SegmentInfo-attribute">
                            <span className="SegmentInfo-attribute-name">{t("card.stakeholders")}</span>
                            <span className="SegmentInfo-attribute-value">
                                <SegmentUsers
                                    segmentId={segmentInfo.id}
                                    allowedToEdit={canEditInfo}
                                    users={segmentInfo.stakeholders}
                                    actionLabel={t("card.addStakeholder")}
                                    addUserFunc={addStakeholderFunc}
                                    removeUserFunc={removeStakeholderFunc}
                                    removeActionLabel={t("card.removeFromStakeholders")}
                                />
                            </span>
                        </div>
                    </div>

                    {exports.map((each) => (
                        <SegmentExport key={segmentInfo.id + each.id} segmentExport={each} />
                    ))}
                    {<AddSegmentExportButton segment={segmentInfo} />}

                    {showAudiencesSegmentFeatures && <AudienceSegmentFeatures />}
                    {tankerModalVisible && (
                        <TankerModal
                            segmentId={segmentInfo.id}
                            nameKey={segmentInfo.tankerNameKey}
                            descriptionKey={segmentInfo.tankerDescriptionKey}
                            defaultName={segmentInfo.name}
                            defaultDescription={segmentInfo.description}
                        />
                    )}
                </div>
            </div>
        )
    );
});

const AddSegmentExportButton = connect((state) => ({
    isAdmin: getAccessState(state).Admin,
    notAllowedSegments: getAccessState(state).RestrictedSegments,
}))((props) => {
    const { isAdmin, notAllowedSegments, segment, dispatch } = props;

    const { t } = useTranslation("segmentExport");

    const handleCreateExport = useCallback(() => {
        if (isAdmin) {
            dispatch(setCreateAdminExportModalVisibility(true));
        } else {
            dispatch(setCreateUserExportModalVisibility(true));
        }
    }, [segment]);

    // TODO: authenticate properly
    const canCreateSegmentExport = isAdmin || (allowedToEditSegment(segment) && !notAllowedSegments);

    const hasExports = segment.exports && !isEmpty(segment.exports.exports);
    const numberOfExports = hasExports ? segment.exports.exports.length : 0;

    return (
        canCreateSegmentExport && (
            <div className={numberOfExports ? "SegmentExport" : "add-first-export-button"}>
                <Button
                    view="clear"
                    theme={numberOfExports ? "clear" : "action"}
                    size={numberOfExports ? undefined : "s"}
                    onClick={handleCreateExport}
                >
                    <FontAwesome name={"plus"} /> {t("new")}
                </Button>
                <CreateAdminSegmentExportModal segmentId={segment.id} />
                <CreateSegmentExportModal />
            </div>
        )
    );
});

export default React.memo(SegmentInfo);
