import React, {useEffect, useState} from "react";
import {useTranslation} from "react-i18next";
import {connect} from "react-redux";
import {Button, Modal, Spin} from "@lib/components/lego";
import LabModalHeader from "../../../utils/LabModalHeader";
import {getActiveIdsState, getApiState, getSegmentGrantsState} from "../../../selectors";
import {
    deleteSegement,
    fetchGrantsForManyAudienceSegments,
} from "../../actions";

import "./style.scss";

function isAudienceSegment(keywordId) {
    return keywordId === 557;
}

const DeleteSegmentDialog = connect((state) => ({
    api: getApiState(state),
    grants: getSegmentGrantsState(state),
    segmentId: getActiveIdsState(state).segment,
}))((props) => {
    const { api, segmentId, exports, visible, onClose, grants, dispatch } = props;

    const [audienceSegmentsIds, setAudienceSegmentsIds] = useState([]);
    const [audienceSegmentsToGrants, setAudienceSegmentsToGrants] = useState(new Map());
    const [loading, setLoading] = useState(false);

    const { t, i18n } = useTranslation("segments");
    const lang = i18n.language;

    const getAudienceSegmentsIds = () => {
        return exports.map((exp) => {
            if (isAudienceSegment(exp.keywordId)) {
                return exp.segmentId;
            }
        }).filter(x => x);
    };

    useEffect(() => {
        setAudienceSegmentsIds(getAudienceSegmentsIds());
    }, []);

    useEffect(() => {
        dispatch(fetchGrantsForManyAudienceSegments(api, audienceSegmentsIds));
    }, [audienceSegmentsIds]);

    useEffect(() => {
        setAudienceSegmentsToGrants(grants);

        if (Object.keys(grants).length) {
            setLoading(Object.values(grants).every(function (value) {
                return value.loading;
            }));
        }
    }, [grants]);

    const saveChanges = () => {
        dispatch(deleteSegement(api, segmentId, lang));
    };

    return (
        <Modal theme="normal" visible={visible} onClose={onClose}>
            <LabModalHeader
                title={t("card.deleteSegment.header") + " " + segmentId}
                hasClose
                onClose={onClose}
            />
            {!loading ? (
                <div className="DeleteSegmentModal-container">
                    <div className="DeleteSegmentModal-disclaimer">{t("card.deleteSegment.disclaimer")}</div>
                    {Object.values(exports).length ? (
                        <div className="DeleteSegmentModal-content">
                            <div className="DeleteSegmentModal-disclaimer">{t("card.deleteSegment.whatDelete")}</div>

                            <table className="DeleteSegmentModal-content-table">
                                <tbody>
                                <tr>
                                    <td className="DeleteSegmentModal-label">{t("card.deleteSegment.export")}</td>
                                    <td className="DeleteSegmentModal-label">{t("card.deleteSegment.audience")}</td>
                                    <td className="DeleteSegmentModal-label">{t("card.deleteSegment.grants")}</td>
                                </tr>
                            {exports.map((each) => (
                                <tr>
                                    <td className="DeleteSegmentModal-elem">{each.id}</td>
                                    {isAudienceSegment(each.keywordId) ? (
                                        <td>{each.segmentId}</td>
                                    ) : (
                                        <td>—</td>
                                    )}
                                    {Object.keys(audienceSegmentsToGrants).includes(each.segmentId.toString()) &&
                                    audienceSegmentsToGrants[each.segmentId.toString()]["grants"].length ? (
                                        <td className="DeleteSegmentModal-elem">
                                            <span>
                                                {Object.values(audienceSegmentsToGrants[each.segmentId.toString()]["grants"]).map((login => (
                                                    <div>{login}</div>
                                                )))}
                                            </span>
                                        </td>
                                    ) : (
                                        <td>—</td>
                                    )}
                                </tr>
                            ))}
                                </tbody>
                            </table>
                        </div>
                    ) : (
                        <div></div>
                    )}
            <div className="DeleteSegmentModal-edit-controls">
                <Button view="pseudo" size="m" onClick={saveChanges} className="DeleteSegmentModal-save-changes-button">
                    {t("card.deleteSegment.delete")}
                </Button>
                <Button view="pseudo" size="m" onClick={onClose}>
                    {t("card.deleteSegment.cancel")}
                </Button>
            </div>
            </div>
            ) : (
                <div className="DeleteSegment-loading">
                    <Spin progress view="default" size="m" />
                </div>
            )}
        </Modal>
    );
});

export default DeleteSegmentDialog;
