import React, { useEffect, useState } from "react";
import { connect } from "react-redux";
import { useTranslation } from "react-i18next";
import { Button, Modal, Textarea, TextInput } from "@lib/components/lego";
import Failable from "../../../utils/Failable";
import LabModalHeader from "../../../utils/LabModalHeader";
import { getNotLocalizedSegmentState, getUpdatedNameAndDescriptionState } from "../../selectors";
import { fetchNotLocalizedSegment, updateNameAndDescription } from "../../actions";
import { getApiState } from "../../../selectors";

import "./style.scss";

function extractProperties(properties, propertyName) {
    let property = { en: "", ru: "" };
    if (properties && properties[propertyName]) {
        let parsed = JSON.parse(properties[propertyName]);
        property.en = parsed.en;
        property.ru = parsed.ru;
    }
    return property;
}

function packProperties(en, ru) {
    return JSON.stringify({ en: en, ru: ru });
}

const EditSegmentDialog = connect((state) => ({
    api: getApiState(state),
    segmentProperties: getNotLocalizedSegmentState(state),
    updatedNameAndDescription: getUpdatedNameAndDescriptionState(state),
}))((props) => {
    const { api, segment, segmentProperties, updatedNameAndDescription, visible, onClose, dispatch } = props;

    const [nameEn, setNameEn] = useState(null);
    const [nameRu, setNameRu] = useState(null);
    const [descriptionEn, setDescriptionEn] = useState(null);
    const [descriptionRu, setDescriptionRu] = useState(null);
    const [initialNameEn, setInitialNameEn] = useState(null);
    const [initialNameRu, setInitialNameRu] = useState(null);
    const [initialDescriptionEn, setInitialDescriptionEn] = useState(null);
    const [initialDescriptionRu, setInitialDescriptionRu] = useState(null);

    const { t, i18n } = useTranslation("segments");
    const lang = i18n.language;

    useEffect(() => {
        dispatch(fetchNotLocalizedSegment(api, segment.id));
    }, []);

    useEffect(() => {
        let name = extractProperties(segmentProperties.value, "name");
        let description = extractProperties(segmentProperties.value, "description");

        setNameEn(name.en);
        setNameRu(name.ru);
        setDescriptionEn(description.en);
        setDescriptionRu(description.ru);

        setInitialNameEn(name.en);
        setInitialNameRu(name.ru);
        setInitialDescriptionEn(description.en);
        setInitialDescriptionRu(description.ru);
    }, [segmentProperties]);

    useEffect(() => {
        dispatch(fetchNotLocalizedSegment(api, segment.id));
    }, [segment]);

    const onSaveChanges = () => {
        if (
            nameRu !== initialNameRu ||
            nameEn !== initialNameEn ||
            descriptionRu !== initialDescriptionRu ||
            descriptionEn !== initialDescriptionEn
        ) {
            dispatch(
                updateNameAndDescription(
                    api,
                    lang,
                    segment.id,
                    packProperties(nameEn, nameRu),
                    packProperties(descriptionEn, descriptionRu)
                )
            );
        }

        onClose();
    };

    const loading = updatedNameAndDescription.loading || segmentProperties.loading;

    return (
        <Modal theme="normal" visible={visible} onClose={onClose}>
            <LabModalHeader
                title={t("card.editProperties.title")}
                onClose={onClose}
                info={segmentProperties.id}
                hasClose
            />

            <Failable failed={segmentProperties.error} errorMessage={t("card.editProperties.errorLoadingSegment")}>
                <div className="EditSegmentDialog">
                    <div className="EditSegmentDialog-segment-name">{segment.name}</div>

                    <div className="EditSegmentDialog-property">
                        <span className="EditSegmentDialog-property-label">{t("card.editProperties.nameRu")}</span>
                        <span className="EditSegmentDialog-property-value">
                            {/*TODO: use withDebounceInput*/}
                            <TextInput
                                className="EditSegmentDialog-property-value"
                                size="s"
                                view="default"
                                value={nameRu}
                                disabled={loading}
                                onChange={(event) => setNameRu(event.target.value)}
                            />
                        </span>
                    </div>

                    <div className="EditSegmentDialog-property">
                        <span className="EditSegmentDialog-property-label">{t("card.editProperties.nameEn")}</span>
                        <span className="EditSegmentDialog-property-value">
                            <TextInput
                                className="EditSegmentDialog-property-value"
                                size="s"
                                view="default"
                                value={nameEn}
                                disabled={loading}
                                onChange={(event) => setNameEn(event.target.value)}
                            />
                        </span>
                    </div>

                    <div className="EditSegmentDialog-property">
                        <span className="EditSegmentDialog-property-label">
                            {t("card.editProperties.descriptionRu")}
                        </span>
                        <span className="EditSegmentDialog-property-value">
                            {/*TODO: use withDebounceInput*/}
                            <Textarea
                                className="EditSegmentDialog-property-value"
                                size="s"
                                view="default"
                                rows={3}
                                value={descriptionRu}
                                disabled={loading}
                                onChange={(event) => setDescriptionRu(event.target.value)}
                            />
                        </span>
                    </div>

                    <div className="EditSegmentDialog-property">
                        <span className="EditSegmentDialog-property-label">
                            {t("card.editProperties.descriptionEn")}
                        </span>
                        <span className="EditSegmentDialog-property-value">
                            <Textarea
                                className="EditSegmentDialog-property-value"
                                size="s"
                                view="default"
                                rows={3}
                                value={descriptionEn}
                                disabled={loading}
                                onChange={(event) => setDescriptionEn(event.target.value)}
                            />
                        </span>
                    </div>

                    <div className="EditSegmentDialog-actions">
                        <Button size="s" view="action" onClick={onSaveChanges} disabled={loading}>
                            {t("card.editProperties.saveChanges")}
                        </Button>
                        <Button size="s" view="default" onClick={onClose} disabled={loading}>
                            {t("card.editProperties.cancel")}
                        </Button>
                    </div>
                </div>
            </Failable>
        </Modal>
    );
});

export default EditSegmentDialog;
