import React, { useEffect, useState } from "react";
import { connect } from "react-redux";
import { useTranslation } from "react-i18next";
import { isEmpty } from "lodash";
import FontAwesome from "react-fontawesome";

import { Button, Modal, Textarea, Spin } from "@lib/components/lego";
import Loading from "@lib/components/loading";
import LabModalHeader from "../../../utils/LabModalHeader";
import { fetchSegment, setTankerModalVisibility } from "../../actions";
import { getTankerDeleteState, getTankerDescriptionState, getTankerNameState, getTankerSaveState } from "./selectors";
import { getActiveSegmentInfoState, getSegmentsModalsState } from "../../selectors";
import {
    deleteTankerKeys,
    fetchDescriptionByTankerKey,
    fetchNameByTankerKey,
    resetTankerModalValues,
    storeTankerValues,
} from "./actions";
import { getApiState } from "../../../selectors";
import TankerKeyView from "./TankerKeyView";

import "./style.scss";

const TankerModal = connect((state) => ({
    api: getApiState(state),
    visible: getSegmentsModalsState(state).tanker,
    name: getTankerNameState(state),
    description: getTankerDescriptionState(state),
    segmentLoading: getActiveSegmentInfoState(state).loading,
    savingInProgress: getTankerSaveState(state).loading,
    deletingInProgress: getTankerDeleteState(state).loading,
}))((props) => {
    const {
        api,
        segmentId,
        nameKey,
        name,
        descriptionKey,
        description,
        defaultName,
        defaultDescription,
        segmentLoading,
        savingInProgress,
        deletingInProgress,
        visible,
        dispatch,
    } = props;

    const [readyToCreate, setReadyToCreate] = useState(false);
    const [readyToDelete, setReadyToDelete] = useState(false);
    const [updatedName, setUpdatedName] = useState("");
    const [updatedDescription, setUpdatedDescription] = useState("");

    const { t, i18n } = useTranslation("segments");
    const language = i18n.language;

    useEffect(() => {
        !isEmpty(nameKey) && dispatch(fetchNameByTankerKey(api, nameKey));
        !isEmpty(descriptionKey) && dispatch(fetchDescriptionByTankerKey(api, descriptionKey));

        if (language === "ru") {
            setUpdatedName(defaultName);
            setUpdatedDescription(defaultDescription);
        }
    }, []);

    const close = () => {
        dispatch(setTankerModalVisibility(false));
        dispatch(resetTankerModalValues());
        dispatch(fetchSegment(api, language, segmentId));
    };

    const createName = () => {
        window.alert("Saving not available now");
    };

    const createDescription = () => {
        window.alert("Saving not available now");
    };

    const saveNameAndDescription = () => {
        dispatch(storeTankerValues(api, segmentId, updatedName, updatedDescription, true)).then((done) => {
            close();
        });
    };

    const deleteKeysAndClose = () => {
        dispatch(deleteTankerKeys(api, segmentId)).then((done) => {
            setReadyToDelete(false);
            close();
        });
    };

    const readyToSave = !isEmpty(updatedName) && !isEmpty(updatedDescription);

    return (
        <Modal theme="normal" visible={visible} onClose={close}>
            <LabModalHeader title={t("tanker.title")} onClose={close} hasClose />

            <div className="TankerModal">
                {isEmpty(nameKey) ^ isEmpty(descriptionKey) || (!isEmpty(nameKey) && !isEmpty(descriptionKey)) ? (
                    <div className="TankerModal-translations">
                        <TankerKeyView
                            tankerKey={nameKey}
                            value={name.value}
                            loading={name.loading}
                            label={t("tanker.name")}
                            createWelcome={t("tanker.create.welcomeName")}
                            createFunc={createName}
                        />
                        <TankerKeyView
                            tankerKey={descriptionKey}
                            value={description.value}
                            loading={description.loading}
                            label={t("tanker.description")}
                            createWelcome={t("tanker.create.welcomeDescription")}
                            createFunc={createDescription}
                        />

                        <div className="TankerModal-delete-keys">
                            {!readyToDelete && (
                                <Button
                                    view="clear"
                                    size="s"
                                    onClick={() => setReadyToDelete(true)}
                                    title={t("tanker.delete")}
                                >
                                    <FontAwesome name="trash" />
                                </Button>
                            )}

                            {readyToDelete && (
                                <div className="TankerModal-delete-keys-action">
                                    <Spin view="default" progress={deletingInProgress} size="s" />
                                    <Button
                                        view="action"
                                        size="s"
                                        onClick={deleteKeysAndClose}
                                        title={t("tanker.delete")}
                                        disabled={deletingInProgress}
                                    >
                                        {t("tanker.delete")}
                                    </Button>
                                </div>
                            )}
                        </div>
                    </div>
                ) : (
                    <div className="TankerModal-create-all">
                        {!readyToCreate && (
                            <div className="TankerModal-create-welcome">
                                <div className="TankerModal-create-welcome-text">{t("tanker.create.welcomeAll")}</div>
                                <Button view="action" size="s" onClick={() => setReadyToCreate(true)}>
                                    {t("tanker.createAll")}
                                </Button>
                            </div>
                        )}

                        {readyToCreate && (
                            <div className="TankerModal-create">
                                <div className="TankerModal-create-form">
                                    <div className="TankerModal-create-form-label">{t("tanker.name")}</div>
                                    <Textarea
                                        view="default"
                                        size="s"
                                        rows={2}
                                        value={updatedName}
                                        onChange={(event) => setUpdatedName(event.target.value)}
                                        placeholder={t("tanker.create.enterValue")}
                                        hasClear
                                    />
                                    <div className="TankerModal-create-form-label">{t("tanker.description")}</div>
                                    <Textarea
                                        view="default"
                                        size="s"
                                        rows={2}
                                        value={updatedDescription}
                                        onChange={(event) => setUpdatedDescription(event.target.value)}
                                        placeholder={t("tanker.create.enterValue")}
                                        hasClear
                                    />

                                    <div className="TankerModal-create-actions">
                                        <Loading loaded={!segmentLoading && !savingInProgress} size="s">
                                            <Button
                                                view="action"
                                                size="s"
                                                onClick={saveNameAndDescription}
                                                disabled={!readyToSave}
                                            >
                                                {t("tanker.create.save")}
                                            </Button>

                                            <Button view="default" size="s" onClick={close}>
                                                {t("tanker.create.close")}
                                            </Button>
                                        </Loading>
                                    </div>
                                </div>
                            </div>
                        )}
                    </div>
                )}
            </div>
        </Modal>
    );
});

export default TankerModal;
